# coding: utf-8

import logging
import json

import yenv
import yt.wrapper as yt

from irt.monitoring.solomon.sensors import SolomonAgentSensorsClient

from bannerland.archive_workers.common import BLYTWorker, get_offer_source_from_feed_url, convert_task_feed_type_str
from bannerland.client.directintapi import DirectIntApiClient
import irt.bannerland.switcher_filter


class SetDirectStatusWorker(BLYTWorker):
    def __init__(self, **kwargs):
        super(SetDirectStatusWorker, self).__init__(**kwargs)
        self.directintapi_client = DirectIntApiClient(yenv.type == yenv.STABLE)

    def do_work(self, pocket_dir):
        logging.info('do_work for %s', pocket_dir)
        order_group_info = self.get_order_group_info(pocket_dir)
        logging.info('to process: %d orders', len(order_group_info))
        if not order_group_info:
            logging.warning('got empty order_group_info')
            return
        result = self.directintapi_client.set_status_for_ad_groups(order_group_info)
        errors = result.get('error_items', {})
        logging.info('set status done; errors: %s', errors)
        self.assert_result(errors)
        self.monitor_result(errors)

    def is_for_grut(self, row):
        task_inf = json.loads(row['task_inf'])
        return irt.bannerland.switcher_filter.is_for_grut(
            order_id=row['OrderID'],
            shop_id=task_inf.get('ShopID'),
            task_type=self.task_type,
            offer_source=get_offer_source_from_feed_url(task_inf.get('Resource', {}).get('FeedUrl')),
            feed_type=convert_task_feed_type_str(task_inf.get('Resource', {}).get('LastValidFeedType', None))
        )

    def get_order_group_info(self, pocket_dir):
        info = {}
        for row in self.yt_client.read_table(yt.ypath_join(pocket_dir, 'tasks.final')):
            if self.is_for_grut(row):
                continue
            OrderID = row['OrderID']
            task_inf = json.loads(row['task_inf'])
            GroupExportID = int(task_inf['GroupExportIDs'][0])
            info.setdefault(OrderID, {})
            # for task_type=dyn there may be many tasks with one group
            info[OrderID].setdefault(GroupExportID, 0)
            if row['banners_count']:
                info[OrderID][GroupExportID] = 1
        return info

    def assert_result(self, errors):
        if len(errors) >= 5:
            raise RuntimeError('Too many orders with errors')

        groups_with_errors = sum(len(v) for _, v in errors.items())
        if groups_with_errors >= 20:
            raise RuntimeError('Too many groups with errors')

    def monitor_result(self, errors):
        groups_with_errors = sum(len(v) for _, v in errors.items())
        solomon_client = SolomonAgentSensorsClient()
        solomon_client.push_single_sensor(
            cluster=self.solomon_cluster,
            service='bannerland_yt',
            sensor='{}.archive_workers.set_direct_status.groups_with_errors'.format(self.task_type),
            value=groups_with_errors,
        )
