# coding: utf-8

import logging

from yt.transfer_manager.client import TransferManager
import yt.wrapper as yt

from bm import yt_tools

from bannerland.archive_workers.common import BLYTWorker


class TransferWorker(BLYTWorker):
    def __init__(self, transfer_manager_token=None, transfer_manager_url=None, **kwargs):
        super(TransferWorker, self).__init__(**kwargs)
        self.tm_client = TransferManager(url=transfer_manager_url, token=transfer_manager_token)

    def get_transfer_pool(self, repl_cluster):
        pool = self.yt_pool
        if (repl_cluster == 'arnold') and isinstance(pool, str) and pool.startswith('bannerland'):
            pool = 'bannerland'
        return pool

    def transfer_node(self, path, copy_attrs=()):
        """Transfer node (table or map_node) and copy attributes."""
        yt_client = self.yt_client
        node_type = yt_tools.get_type(path, yt_client=yt_client)
        if node_type == 'table':
            self.transfer([path])
        elif node_type == 'map_node':
            tables = yt_client.list(path, absolute=True)
            self.transfer(tables)
            for attr in copy_attrs:
                value = yt_tools.get_attribute(path, attr, yt_client=yt_client)
                for repl in self.get_replicas():
                    yt_tools.set_attribute(path, attr, value, yt_client=repl['client'])
        else:
            raise ValueError("Node type not supported: {}".format(node_type))

    def transfer(self, tables):
        src_cluster = self.yt_cluster
        for repl in self.get_replicas():
            # tm сам создаст директории, но не будем на это закладываться; заодно проверим работу клиента
            for tbl in tables:
                target_yt_client = repl['client']
                target_yt_client.create('map_node', yt.ypath_dirname(tbl), recursive=True, ignore_existing=True)
                self.tm_client.add_task(
                    src_cluster, tbl,
                    repl['cluster'], tbl,
                    sync=True,
                    params={
                        'queue_name': 'default',  # пока отдельная очередь не нужна
                        'pool': self.get_transfer_pool(repl['cluster']),
                    },
                )
                # transfer manager should copy user attrs!
                logging.info('transfered %s: %s -> %s', tbl, src_cluster, repl['cluster'])

    def get_replicas(self):
        if not hasattr(self, '__replicas'):
            self.__replicas = []
            for repl_cluster in self.get_cypress_config().repl_clusters:
                client = yt.YtClient(proxy=repl_cluster, config={'token_path': self.yt_client.config['token_path']})
                self.__replicas.append({'cluster': repl_cluster, 'client': client})
            return self.__replicas
