# coding: utf-8

import json
import logging
import requests
from collections import namedtuple, Counter
from multiprocessing import Pool

import yenv

from six.moves.urllib.parse import quote

import irt.bannerland.options

from bm.yt_tools import DynTableCache, CacheMode

from irt.tvm import TVM, TVMService
from bannerland.client.utils import make_http_requests


# NOTE (fawkes@): 457 days ≈ 1 year (Ava ttl) + 3 month (safe period for a banner deleting)
DEFAULT_AVA_TTL = '457d'


# информация о загруженном в аватарницу изображении
class UploadedImage(namedtuple('UploadedImage', ['group_id', 'imagename'])):

    @classmethod
    def from_url(cls, url):
        # url example: //avatars.mds.yandex.net/get-yabs_performance/2037436/2a0000016eef5c0b9731c8eb43f6c872d55d/huge
        url_parts = url.split('/')
        group_id = int(url_parts[-3])
        imagename = url_parts[-2]
        return cls(group_id, imagename)


class AvatarsClient:
    @classmethod
    def init_by_task_type(cls, task_type='perf', **kwargs):
        if task_type not in ['dyn', 'perf']:
            raise ValueError('task_type = %s not found', task_type)

        if 'ns' not in kwargs:
            task_type2ns = {
                'dyn': 'yabs_performance',
                'perf': 'yabs_performance',
            }
            kwargs['ns'] = task_type2ns[task_type]

        if 'cache_dyntable' not in kwargs:
            if yenv.type != yenv.STABLE:
                kwargs['cache_dyntable'] = '//home/bannerland/test/perf_avatars_cache'
            else:
                task_type2cache_dyntable_option_name = {
                    'dyn': 'perf_avatars_cache_dyntable',
                    'perf': 'perf_avatars_cache_dyntable',
                }
                kwargs['cache_dyntable'] = irt.bannerland.options.get_option(task_type2cache_dyntable_option_name[task_type])

        kwargs['need_tvm'] = False  # tmv need if we create new ava ns
        kwargs.setdefault('solomon_sensor_prefix', '{}.'.format(task_type))
        ava_client = AvatarsClient(**kwargs)
        return ava_client

    # yt_client нужен для работы с кэшом (как точечные запросы, так и map-reduce)
    def __init__(
        self,
        put_images_pack_size=1000, avatars_process_count=4, init_process_pool=False,
        avatars_retries=5,
        ns='yabs_performance', need_tvm=False,
        cache_dyntable=None, read_cache=CacheMode.NEVER, write_cache=CacheMode.ALWAYS,
        lookup_client=None, map_reduce_client=None, solomon_client=None, solomon_sensor_prefix='', ttl=DEFAULT_AVA_TTL,
    ):
        if cache_dyntable is None:
            raise ValueError('cache_dyntable not selected')

        self.tvm_service = None
        self.need_tvm = need_tvm

        if lookup_client is None:
            raise RuntimeError('No lookup yt client')

        if map_reduce_client is None:
            raise RuntimeError('No map_reduce yt client')

        if map_reduce_client.config["proxy"]["url"] == 'markov':
            raise RuntimeError('Attempt to use `markov` as map reduce cluster')

        self.lookup_client = lookup_client
        self.map_reduce_client = map_reduce_client
        if yenv.type != yenv.STABLE:
            self.host = 'avatars-int.mdst.yandex.net'
            self.read_host = 'avatars.mdst.yandex.net'
        else:
            self.host = 'avatars-int.mds.yandex.net'
            self.read_host = 'avatars.mds.yandex.net'  # для формирования урлов

        self.rate_limit_code = 429  # http-код ответа аватарницы в случае превышения лимита

        self.read_cache = read_cache
        self.write_cache = write_cache
        self.ttl = ttl

        self.namespace = ns
        self.base_url = 'http://{}:13000'.format(self.host)
        self.read_url = 'http://{}'.format(self.read_host)
        self._put_req_url = '{}/put-{}'.format(self.base_url, self.namespace)

        self.cache = DynTableCache(
            table=cache_dyntable,
            key_field='url',
            value_fields=['avatars', 'meta', 'response_code'],
            check_fields=['avatars'],
            ttl=3600 * 24 * 365,
            lookup_client=self.lookup_client,
            map_reduce_client=self.map_reduce_client,
        )

        self.put_images_pack_size = put_images_pack_size
        self.avatars_process_count = avatars_process_count

        if init_process_pool:
            self.process_pool = Pool(self.avatars_process_count)
        else:
            self.process_pool = None

        self.avatars_retries = avatars_retries
        self.avatars_chunksize = 1  # сколько урлов заливает один процесс

        self.solomon_client = solomon_client
        self.solomon_sensor_prefix = solomon_sensor_prefix

    def __del__(self):
        if self.process_pool is not None:
            self.process_pool.close()
            self.process_pool.join()

    @staticmethod
    def get_avatars_tvm_service():
        tvm = TVM()
        if yenv.type != yenv.STABLE:
            tvm_service = TVMService(tvm, service='test_avatars')
        else:
            tvm_service = TVMService(tvm, service='prod_avatars')
        return tvm_service

    def get_requests_by_urls(self, urls):
        if not self.need_tvm:
            request_f = requests.Request
        else:
            tvm_service = self.get_avatars_tvm_service()
            request_f = tvm_service.Request

        # Urls must be punicode earlier
        # Urls could quoted by client, so safe parameters was added (double quote doesn't break url)

        return ((url, request_f('GET', self._put_req_url, params={'url': quote(url, safe=':/?%&='), 'expire': self.ttl})) for url in urls)  # generator; req_id = url

    # залить картинки в аватарницу по урлу (ручка /put), возвращает code и dict ответа аватарницы
    def put_image_urls(self, urls):
        logging.info('put_image_urls ...')
        prepared_requests = self.get_requests_by_urls(urls)
        gen_resps = make_http_requests(  # also generator
            prepared_requests, process_count=self.avatars_process_count, process_pool=self.process_pool,
            chunksize=self.avatars_chunksize,
            rate_limit_code=self.rate_limit_code, retries=self.avatars_retries, parse_json=True,
            timeout=30,
        )
        result = {url: resp for url, resp in gen_resps}
        logging.info('put_image_urls done!')
        return result

    # запросы для изображений, хранящихся в аватарнице
    # method  -  delete|getinfo
    # images  -  list of UploadedImage's or just tuples (group_id, imagename)
    # path  -  if not None, appended to url (assumed to contain leading slash!)
    # use_read_host - option to use read_url
    # request_type - GET|HEAD
    def request_for_images(self, method, images, path=None, use_read_host=False, request_type='GET', **kwargs):
        if use_read_host:
            base_url = self.read_url
        else:
            base_url = self.base_url

        def gen_reqs():
            for image in images:
                image = UploadedImage(*image)
                parts = [base_url, method + '-' + self.namespace, str(image.group_id), image.imagename]
                url = '/'.join(parts)
                if path is not None:
                    url = url + path
                req = requests.Request(request_type, url)
                yield image, req

        gen_resps = make_http_requests(
            gen_reqs(), process_count=self.avatars_process_count, process_pool=self.process_pool,
            rate_limit_code=self.rate_limit_code, retries=self.avatars_retries, parse_json=True, **kwargs
        )
        return {image: resp for image, resp in gen_resps}

    # удаляем неиспользуемые тяжёлые поля из мета-информации
    def format_meta(self, resp_data):
        meta = resp_data.get('meta')
        if meta is None:
            return meta
        meta.pop('SmartCrop', None)
        meta.pop('SmartCropCompactSaliency', None)
        return meta

    @classmethod
    def check_avatars(cls, ava):
        required_sizes = ['small', 'big', 'huge']
        if any(size not in ava for size in required_sizes):
            return False
        return True

    # формат для хранения аватарок в кэше
    def format_avatars(self, resp_data):
        ava = {}
        required_fields = set(['path', 'width', 'height'])
        add_fields = ['smart-centers', 'smart-center']
        for size, info in resp_data.get('sizes', {}).items():
            if any(f not in info for f in required_fields):
                continue
            bs_url = '//' + self.read_host + info['path']  # path already has a leading slash
            current_ava = {
                'width': info['width'],
                'height': info['height'],
                'url': bs_url,
            }
            for f in add_fields:
                if f in info:
                    current_ava[f] = info[f]
            ava[size] = current_ava

        if not self.check_avatars(ava):
            return None

        return json.dumps(ava, separators=(',', ':'))  # avoid whitespace for compactness

    # по списку урлов получить dict {url: {"avatars": ava, "meta": meta, "response_code": code}}
    # ava и/или meta могут быть None (битый урл, некорректный ответ аватарницы)
    def get_avatars(self, urls):
        return self.cache.call(self._get_avatars, urls, read_cache=self.read_cache, write_cache=self.write_cache)

    def _get_avatars(self, urls):
        url2res = self.put_image_urls(urls)
        url2ava = {}
        for url, res in url2res.items():
            if res is None:
                # не сумели получить ни одного http request  ->  не кэшируем
                continue
            resp_code, resp_json = res['response_code'], res['response_json']
            if resp_code == 200 and resp_json is not None:
                avatars = self.format_avatars(resp_json)
                meta = self.format_meta(resp_json)
            elif resp_code == self.rate_limit_code or resp_code >= 500:
                # не сумели получить ни положительного результата, ни внятного кода ошибки  ->  не кэшируем
                continue
            else:
                avatars = None
                meta = None
            url2ava[url] = {'response_code': resp_code, 'avatars': avatars, 'meta': meta}

        self.monitor_avatars_result(urls, url2ava)
        return url2ava

    # к таблице с полем "url" доклеить поля "avatars", "meta", "response_code" (см. get_avatars)
    def get_avatars_for_table(self, input_table, output_table, **kwargs):
        return self.cache.call_for_table(self.get_avatars, input_table, output_table, call_pack_size=self.put_images_pack_size, **kwargs)

    def monitor_avatars_result(self, input, output):
        if self.solomon_client is None:
            return
        sensors = {}
        sensors['put_requests_count'] = len(input)
        sensors['put_responses_lost_count'] = len(input) - len(output)
        counts = Counter()
        for ava in output.values():
            if ava['avatars']:
                result = 'success'
            else:
                result = 'fail{}'.format(ava['response_code'])
            counts[result] += 1

        for result, count in counts.items():
            name = 'put_responses_{}_count'.format(result)
            sensors[name] = count

        cluster = self.map_reduce_client.config['proxy']['url'].split('.')[0]
        for name, val in sensors.items():
            self.solomon_client.push_single_sensor(
                cluster='yt_{}'.format(cluster),
                service='bannerland_yt',
                sensor='{}avatars.{}'.format(self.solomon_sensor_prefix, name),
                value=val,
            )
