# coding: utf-8

import json
import logging

import requests

from bannerland.client.utils import make_http_request

DIRECT_API_URL_STABLE = 'http://intapi.direct.yandex.ru/jsonrpc/BannerLand'
DIRECT_API_URL_TESTING = 'http://intapi.test.direct.yandex.ru:9000/jsonrpc/BannerLand'


class DirectIntApiClient:
    def __init__(self, is_production):
        self.url = DIRECT_API_URL_STABLE if is_production else DIRECT_API_URL_TESTING

        logging.info('init directintapi client, url: %s', self.url)

    def jsonrpc_call(self, method, data):
        request = requests.Request('POST', url=self.url, data=json.dumps({'method': method, 'params': data}))
        result = make_http_request(request, parse_json=True)
        if not result:
            raise ValueError("Can't get valid response")
        if result.get('response_code') != 200:
            raise ValueError("Can't get 200 OK response")
        if 'response_json' not in result:
            raise ValueError("Can't get json response")
        return result['response_json']

    # https://st.yandex-team.ru/DYNSMART-1308
    # https://wiki.yandex-team.ru/testirovanie/functesting/direkt/automatization/intapi/manual/BannerLand/
    def set_status_for_ad_groups(self, order_group_info):
        """Set generation statuses for groups.

        Input data format: dict {OrderID: {GroupExportID: NonEmptyFlag}}
        Flag is interpreted as boolean:
            True = generated >0 banners for group, maps to "non-empty"
            False = generated 0 banners for group, maps to "empty"

        Returns list of error items.
        """
        method = 'setStatusForAdGroupsWithOrderID'

        status = {}
        for OrderID, group_info in order_group_info.items():
            status[OrderID] = {GroupExportID: ('non-empty' if Flag else 'empty') for GroupExportID, Flag in group_info.items()}

        def gen_packs(data):
            pack_size = 1000  # ограничиваем кол-во групп
            curr_size = 0
            pack = {}
            for key, sub_data in data.items():
                pack[key] = sub_data
                curr_size += len(sub_data)
                if curr_size >= pack_size:
                    yield pack
                    pack = {}
                    curr_size = 0
            if pack:
                yield pack

        error_items = {}
        for pack in gen_packs(status):
            resp = self.jsonrpc_call(method, pack)
            result = resp['result']
            if not result['success']:
                # success=0 - грубая ошибка (невалидный запрос?), такого быть не должно
                raise ValueError('Invalid request: got success=0')
            for k, v in result.get('error_items', {}).items():
                error_items[k] = v

        return {'error_items': error_items}
