# coding: utf-8

import time
import logging
from multiprocessing import Pool
import requests


# retry only opts['rate_limit_code'] or 5xx code
def _make_http_request(args):
    req_id, req, opts = args
    prep_req = req.prepare()
    session = requests.Session()
    result, resp, code = None, None, -1
    retries = opts.get('retries', 1)
    timeout = opts.get('timeout', 60)
    get_headers = opts.get('get_headers', False)
    logging.debug('_make_http_request: will make request (%s, %s)', req.url, req.params)

    for n_try in range(retries):
        time_to_sleep = 1 + 2 * n_try if n_try != retries - 1 else 0
        try:
            resp = session.send(prep_req, timeout=timeout)
            code = resp.status_code
            logging.debug('got response code %d, content: %s', code, resp.content)
            result = {'response_code': code}
            if opts.get('parse_json'):
                try:
                    data = resp.json()
                except:
                    data = None
                result['response_json'] = data
            else:
                result['response_content'] = resp.content
            if get_headers:
                result['headers'] = resp.headers

            if code == opts.get('rate_limit_code'):
                logging.warning('_make_http_request: hit rate limit, sleep time doubles!')
                time_to_sleep *= 2
            elif code < 500:
                break  # will return this resp
        except Exception as e:
            logging.warning('_make_http_request: got exception: %s', e)
            code = -1

        # TODO: log req.data, but limit length
        logging.warning(
            '_make_http_request: request (%s, %s) failed with code %d; try %d of %d',
            req.url, req.params,
            code, n_try + 1, retries,
        )
        time.sleep(time_to_sleep)

    if code != 200:
        logging.warning(
            'request (%s, %s) failed, last response: code %d, content: %s',
            req.url, req.params,
            code, (resp.content if resp is not None else None),
        )

    return req_id, result


# params:
#   input is an iterable over pairs (req_id, req), req is http request constructed via requests library
#       (req_id is needed, because output order is not preserved)
#   process_count - number of http request processes
#   process_pool - do not create new pool, use given
# other params are passed to _make_http_request:
#   parse_json: try to parser output, result with 'response_json' field instead of 'response_content'
#   retries: subj
#   rate_limit_code: special http code, sleep twice if got it
# generates pairs (req_id, result)
def make_http_requests(input, process_count=1, chunksize=1, parse_json=False, retries=1, rate_limit_code=None, process_pool=None, get_headers=False, timeout=60):
    opts = {
        'parse_json': parse_json,
        'retries': retries,
        'rate_limit_code': rate_limit_code,
        'get_headers': get_headers,
        'timeout': timeout,
    }
    created_pool = False
    if process_pool is None:
        process_pool = Pool(process_count)
        created_pool = True
    args_gen = ((req_id, req, opts) for req_id, req in input)
    for req_id, result in process_pool.imap_unordered(_make_http_request, args_gen, chunksize=chunksize):
        yield req_id, result
    if created_pool:
        process_pool.close()
        process_pool.join()
    return


def make_http_request(http_request, **kwargs):
    args = (None, http_request, kwargs)
    _, response = _make_http_request(args)
    return response
