#!/usr/bin/env python
# -*- coding: utf-8 -*-

import os
import argparse
import logging

import yt.wrapper as yt

from bannerland.yt_local import get_proxy as get_local_proxy
from bannerland.make_banners import get_maker_cls


def main():
    argparser = argparse.ArgumentParser(formatter_class=argparse.ArgumentDefaultsHelpFormatter)

    argparser.add_argument('--yt-dir', help='yt working directory')
    argparser.add_argument('--input-tao-table', help='joined tasks and offers')

    argparser.add_argument('--task-type', choices=['dyn', 'perf'], default='perf', help='subj')

    argparser.add_argument(
        '--external-dir', default=None,
        help='dir with ext tables with "external_source_info" attr, see fetch_external_sources cmd in tools; pass empty string to disable',
    )
    argparser.add_argument('--ensure-external', help='create empty external tables if not exist', action='store_true')

    argparser.add_argument('--catalogia', '-C', help='"local": use local prepared catalogia; "sb": use resources from Sandbox', default='sb')
    argparser.add_argument('--fake-archives', action='store_true', help='pack symlinks')

    argparser.add_argument('--mode', choices=['local'], default=None, help='"local" for local yt (implies no YQL, 1 cpu, etc)')

    argparser.add_argument('--start', help='start step', default='begin')
    argparser.add_argument('--stop',  help='stop step (not including!)', default='end')
    argparser.add_argument('--skip-add-avatars', help='skip add_avatars step', action='store_true')

    args = argparser.parse_args()

    cls_args = dict(vars(args))

    start = cls_args.pop('start')
    stop = cls_args.pop('stop')
    task_type = cls_args.pop('task_type')
    ensure_external = cls_args.pop('ensure_external')

    # get some args from environment
    if 'BANNERLAND_CLEANUP_LEVEL' in os.environ:
        cls_args['cleanup_level'] = int(os.environ.get('BANNERLAND_CLEANUP_LEVEL'))

    yt_meta_spec = {}
    for key_name, env_name in {'max_cpu': 'BMYT_MAX_CPU', 'max_mem': 'BMYT_MAX_MEM', 'bm_layers': 'BMYT_LAYERS'}.items():
        if env_name in os.environ:
            yt_meta_spec[key_name] = int(os.environ.get(env_name))
    cls_args['yt_meta_spec'] = yt_meta_spec

    res_type = cls_args.pop('catalogia')
    cls_args['catalogia_spec'] = {
        'lib': {'type': res_type},
        'dicts': {'type': res_type},
        'fake_archives': cls_args.pop('fake_archives'),
    }

    if cls_args.get('external_dir') is not None:
        ext_dir = cls_args.pop('external_dir')
        ext_srcs = []
        # we need yt client here :(
        if args.mode == 'local':
            yt_client = yt.YtClient(proxy=get_local_proxy())
        else:
            yt_client = yt
        if ext_dir:
            for ext_tbl in yt_client.list(ext_dir, absolute=True):
                ext_src = yt_client.get_attribute(ext_tbl, 'external_source_info')
                ext_src['table'] = ext_tbl
                ext_srcs.append(ext_src)
        cls_args['external_sources'] = ext_srcs

    # dont pass None to use meaningful defaults in Maker
    cls_args = {k: v for k, v in cls_args.items() if v is not None}

    logging.warn('args for %s maker: %s', task_type, cls_args)
    maker_cls = get_maker_cls(task_type)
    maker = maker_cls(**cls_args)

    if ensure_external:
        for ext_src in maker.external_sources:
            ext_tbl = ext_src['table']
            if not maker.yt_client.exists(ext_tbl):
                maker.yt_client.create('table', ext_tbl, recursive=True)

    maker.run(start, stop)

if __name__ == "__main__":
    main()
