#!/usr/bin/env python
# -*- coding: utf-8 -*-

import time

import yt.wrapper as yt

import irt.iron.options as iron_opts

from irt.bannerland.options import get_option as get_bl_opt, get_cypress_config


def get_yt_client():
    config = {
        'proxy': {'url': 'hahn'},
        'token_path': iron_opts.get('yt_token_path'),
    }
    return yt.YtClient(config=config)


def get_sensors_prepare_input(task_type):
    data = []
    yt_client = get_yt_client()
    data_dir = get_cypress_config(task_type).get_path('tao_current')
    tasks_and_offers = yt_client.list(data_dir)

    def get_tao_time(tao_table):
        tao_name = tao_table.split('/')[-1]
        return time.mktime(time.strptime(tao_name, 'merged_tasks_and_offers_%Y-%m-%d_%H:%M:%S'))

    if tasks_and_offers:
        data.append({
            'sensor': 'last_tasks_and_offers_hours_old',
            'value': (time.time() - get_tao_time(max(tasks_and_offers))) / 3600,
        })
    day_ago_time = time.time() - 3600 * 24
    day_rows = 0
    for tbl in tasks_and_offers:
        tbl_time = time.mktime(time.strptime(tbl, 'merged_tasks_and_offers_%Y-%m-%d_%H:%M:%S'))
        if tbl_time < day_ago_time:
            continue
        day_rows += yt_client.row_count(data_dir + '/' + tbl)
    data.append({
        'sensor': 'tasks_and_offers_rows_per_day',
        'value': day_rows,
    })

    ava_last_put = yt_client.get_attribute(data_dir, 'last_avatars_put_table')
    data.append({
        'sensor': 'avatars_cache_hours_old',
        'value': (time.time() - get_tao_time(ava_last_put)) / 3600,
    })

    if task_type == 'dyn':
        # avatars cache table is the same, we do not duplicate sensors
        return data

    data.append({
        'sensor': 'avatars_cache_rows',
        # для динтаблиц не работает row_count()
        'value': yt_client.get_attribute(get_bl_opt('perf_avatars_cache_dyntable'), 'chunk_row_count'),
    })

    return data


def get_sensors_make_banners(task_type):
    data = []
    yt_client = get_yt_client()
    conf = get_cypress_config(task_type)
    make_banners_dir = conf.get_path('make_banners_dir')
    arc_dir = conf.get_path('make_banners_archive')
    archives = yt_client.list(arc_dir)
    if archives:
        last_arc = max(archives)
        last_start_time = yt_client.get(arc_dir + '/' + last_arc + '/@start_time')
        last_time = time.mktime(time.strptime(last_start_time, '%Y-%m-%d_%H:%M:%S'))

        error_in_last_pocket = int(bool(yt_client.get_attribute(make_banners_dir, 'last_crash', False)))
        data.append({
            'sensor': 'error_in_last_pocket',
            'value': error_in_last_pocket,
        })
        data.append({
            'sensor': 'last_archive_hours_old',
            'value': (time.time() - last_time) / 3600,
        })

    return data


def get_sensors_archive_workers(task_type):
    yt_client = get_yt_client()
    cypress_conf = get_cypress_config(task_type)
    perf_pocket_workers = ['funnel', 'full_state']
    dyn_pocket_workers = ['funnel', 'full_state']
    config = {
        'pocket': {
            'archive_dir': cypress_conf.get_path('make_banners_archive'),
            'worker_names': (perf_pocket_workers if task_type == 'perf' else dyn_pocket_workers),
        },
        'full_state': {
            'archive_dir': cypress_conf.get_path('full_state_archive'),
            'worker_names': ['publish_fs'],
        }
    }
    data = []
    for worker_type, type_config in config.items():
        archive_workers = type_config['worker_names']
        to_process = {w: 0 for w in archive_workers}
        seen_workers = set()
        arc_dir = type_config['archive_dir']
        dirs = sorted(yt_client.list(arc_dir), reverse=True)
        for arc in dirs:
            for worker_name in archive_workers:
                if worker_name in seen_workers:
                    continue
                if yt_client.get_attribute(arc_dir + '/' + arc, 'archive_worker_' + worker_name, default=None):
                    seen_workers.add(worker_name)
                else:
                    to_process[worker_name] += 1

        for worker_name, count in to_process.items():
            data.append({
                'sensor': worker_name + '_todo_count',
                'value': count,
            })
    return data
