# coding=utf-8
from __future__ import division
import logging
import time
from datetime import datetime, timedelta
from sandbox.projects.common import solomon

import yt.wrapper as yt
import irt.common.yt

import ads.libs.yql as yql_tools
from .queries import yql_aggregate_clients_stats, yql_active_orders, yql_append_domain, yql_combine_tops, yql_final_top, \
    yql_update_old_top, yql_top_clients, yql_total_clients_expenses, combine_queries, yql_preprocess_full_state


MONITOR_NAME = 'top-clients-monitor'
LOCK_NAME_PREFIX = 'bannerland-top-clients'
HOUR_LOG_PATH = '//logs/bs-chevent-log/1h'
TARGET_DOMAINS_TABLE = '//home/yabs/dict/DomainIdToDomain'
HOUR_LOG_NAME_FORMAT = '%Y-%m-%dT%H:%M:%S'
FS_BANNERS_TABLE = 'banners'
FS_ID_ATTRIBUTE = 'FullStateID'
MONITORING_PATH = '//home/bannerland/{}/monitorings/top_clients'
MAX_DAYS_OFF_TOP = 3
MIN_COST_RATIO = 0.02
BANNERS_TYPE_CONFIG = {
    'perf': {
        'context_type': '8',
        'clients_top_table': 'top_clients_perf',
        'lock_table': 'lock_perf',
        'full_state_dir': '//home/bannerland/perf/full_state/export/v3',
        'active_tasks': '//home/blrt/task/perf/export_latest_state/task_export',
        'required_clients': [
            {
                'domain_id': 1281524,
                'domain': 'booking.com'
            }
        ]
    },
    'dyn': {
        'context_type': '7',
        'clients_top_table': 'top_clients_dyn',
        'lock_table': 'lock_dyn',
        'full_state_dir': '//home/bannerland/dyn/full_state/export/v3',
        'active_tasks': '//home/blrt/task/dyn/export_latest_state/task_export',
        'required_clients': [
            {
                'domain_id': 1281524,
                'domain': 'booking.com'
            }
        ]
    },
}


def do_yql(client, query, yt_pool):
    request = yql_tools.run_yql_query(bytes(query, 'utf-8'), client=client, pool=yt_pool, is_async=True)
    logging.info('YQL operation started: %s', request.share_url)
    yql_tools.wait_yql_request(request)


def timestamp(date):
    return time.mktime(date.timetuple())


def get_ts_datetime(dir_time, time_format, seconds_delta=0):
    """
    :return: datetime сенсора в формате Соломона
    """
    ts_datetime = datetime.strptime(dir_time, time_format)
    ts_datetime += timedelta(0, time.timezone + seconds_delta)
    return ts_datetime.strftime('%Y-%m-%dT%H:%M:%SZ')


class TopClientsMonitor(object):
    def __init__(self, yt_token, yql_token, solomon_token, banners_type, yt_cluster, yt_pool, release_type):
        self.monitoring_path = MONITORING_PATH.format(release_type)
        self.active_tasks = BANNERS_TYPE_CONFIG[banners_type]['active_tasks']
        self.clients_top_table = BANNERS_TYPE_CONFIG[banners_type]['clients_top_table']
        self.lock_table = BANNERS_TYPE_CONFIG[banners_type]['lock_table']
        self.required_clients = BANNERS_TYPE_CONFIG[banners_type]['required_clients']
        self.context_type = BANNERS_TYPE_CONFIG[banners_type]['context_type']

        self.solomon_token = solomon_token
        self.banners_type = banners_type
        self.yt_cluster = yt_cluster
        self.yt_pool = yt_pool
        self.release_type = release_type

        yt_config = {
            'token': yt_token,
            'spec_defaults': {},
        }

        if self.yt_pool:
            yt_config['spec_defaults']['pool'] = self.yt_pool

        self.yt_client = yt.YtClient(proxy=self.yt_cluster, config=yt_config)
        self.yql_client = yql_tools.get_yql_client(db=self.yt_cluster, token=yql_token)

        full_state_path = self.yt_client.get(BANNERS_TYPE_CONFIG[banners_type]['full_state_dir'], attributes=['path']).attributes.get('path')
        self.fs_id = irt.common.yt.get_attribute(full_state_path, FS_ID_ATTRIBUTE, yt_client=self.yt_client)
        self.fs_banners_path = yt.ypath_join(full_state_path, FS_BANNERS_TABLE)
        self.preprocessed_full_state = yt.ypath_join(self.monitoring_path, '_preprocessed_fs_{}'.format(self.banners_type))

    def run(self):
        logging.info('Start for %s banners', self.banners_type)
        processed_log_table = self.process_log()
        if processed_log_table:
            sensors = self.get_clients_sensors(processed_log_table)
            params = self._get_solomon_params()
            solomon.push_to_solomon_v2(self.solomon_token, params, sensors, common_labels=(('banners_type', self.banners_type),))
            attribute_name = 'last_processed_{}'.format(self.banners_type)
            irt.common.yt.set_attribute(self.monitoring_path, attribute_name, processed_log_table, self.yt_client)
        logging.info('Finish for %s banners', self.banners_type)

    def process_log(self):
        hour_log_table_name = self._get_hour_log_table()
        if not hour_log_table_name:
            logging.info('There are no new tables to process')
            return

        hour_log_table_path = yt.ypath_join(HOUR_LOG_PATH, hour_log_table_name)
        logging.info('Process hour log table: %s', hour_log_table_path)

        with self.yt_client.TempTable() as clients_stats, \
                self.yt_client.TempTable() as total_expenses, \
                self.yt_client.TempTable() as filtered_old_top, \
                self.yt_client.TempTable() as active_orders, \
                self.yt_client.TempTable() as current_top, \
                self.yt_client.TempTable() as combined_top, \
                self.yt_client.TempTable() as top_with_domains:

            if self._is_need_preprocess_fs:
                do_yql(self.yql_client, yql_preprocess_full_state(self.fs_banners_path, self.preprocessed_full_state), yt_pool=self.yt_pool)
                self.yt_client.set(yt.ypath_join(self.preprocessed_full_state, '@original_fs'), self.fs_id)

            agr_stats_query = yql_aggregate_clients_stats(hour_log_table_path, clients_stats, self.context_type)
            active_orders_query = yql_active_orders(self.preprocessed_full_state, self.active_tasks, active_orders)
            do_yql(self.yql_client, combine_queries([agr_stats_query, active_orders_query]), yt_pool=self.yt_pool)
            do_yql(self.yql_client, yql_total_clients_expenses(clients_stats, total_expenses), yt_pool=self.yt_pool)

            clients_expenses = 0
            for row in self.yt_client.read_table(total_expenses):
                clients_expenses = row['total']
            if clients_expenses == 0:
                raise ValueError('Clients expenses can not be equal to zero')

            cur_log_dt = datetime.strptime(hour_log_table_name, HOUR_LOG_NAME_FORMAT)
            cur_log_timestamp = int(timestamp(cur_log_dt))
            fixed_top_clients = self._get_fixed_clients()
            top_clients_table_path = yt.ypath_join(self.monitoring_path, self.clients_top_table)

            top_clients_query = yql_top_clients(clients_stats, current_top, clients_expenses, fixed_top_clients, MIN_COST_RATIO, cur_log_timestamp)
            do_yql(self.yql_client, top_clients_query, yt_pool=self.yt_pool)

            if self.yt_client.exists(top_clients_table_path):
                update_old_top_q = yql_update_old_top(top_clients_table_path, clients_stats, filtered_old_top, clients_expenses,
                                                      MAX_DAYS_OFF_TOP, MIN_COST_RATIO, cur_log_timestamp)
                do_yql(self.yql_client, update_old_top_q, yt_pool=self.yt_pool)
                do_yql(self.yql_client, yql_combine_tops(filtered_old_top, current_top, combined_top), yt_pool=self.yt_pool)
            else:
                combined_top = current_top

            do_yql(self.yql_client, yql_append_domain(combined_top, TARGET_DOMAINS_TABLE, top_with_domains), yt_pool=self.yt_pool)
            do_yql(self.yql_client, yql_final_top(top_with_domains, active_orders, top_clients_table_path), yt_pool=self.yt_pool)

        return hour_log_table_name

    def get_clients_sensors(self, processed_log_table):
        top_clients_table = yt.ypath_join(self.monitoring_path, self.clients_top_table)
        stat_time_str = processed_log_table
        sensors = []
        for row in self.yt_client.read_table(top_clients_table):
            client_domain = row['Domain']
            base_sensor_name = 'top_clients_stat'
            sensors.append(self._get_sensor('{}.{}'.format(base_sensor_name, 'cost'), int(row['cost']), client_domain, stat_time_str))
            sensors.append(self._get_sensor('{}.{}'.format(base_sensor_name, 'cost'), int(row['cost']), client_domain, stat_time_str, seconds_delta=1800))
            seen_active_ratio = 0
            if row['active_orders_count'] > 0:
                seen_active_ratio = row['seen_orders_count'] / row['active_orders_count']
            sensors.append(self._get_sensor('{}.{}'.format(base_sensor_name, 'seen_active_ratio'), seen_active_ratio, client_domain, stat_time_str))
            sensors.append(self._get_sensor('{}.{}'.format(base_sensor_name, 'seen_active_ratio'), seen_active_ratio, client_domain, stat_time_str, seconds_delta=1800))

            sensors.append(self._get_sensor('{}.{}'.format(base_sensor_name, 'children_count'), int(row['children_count']), client_domain, stat_time_str))
            sensors.append(self._get_sensor('{}.{}'.format(base_sensor_name, 'children_count'), int(row['children_count']), client_domain, stat_time_str, seconds_delta=1800))

            sensors.append(self._get_sensor('{}.{}'.format(base_sensor_name, 'phrases_count'), int(row['phrases_count']), client_domain, stat_time_str))
            sensors.append(self._get_sensor('{}.{}'.format(base_sensor_name, 'phrases_count'), int(row['phrases_count']), client_domain, stat_time_str, seconds_delta=1800))
        return sensors

    def _get_solomon_params(self):
        return {
            'project': 'bannerland',
            'cluster': self.release_type,
            'service': 'top_clients',
        }

    def _get_sensor(self, sensor_name, value, client_domain, time_str, seconds_delta=0):
        return {
            'ts': get_ts_datetime(time_str, HOUR_LOG_NAME_FORMAT, seconds_delta),
            'value': value,
            'labels': {
                'sensor': sensor_name,
                'client': client_domain,
            },
        }

    def _get_hour_log_table(self):
        attribute_name = 'last_processed_{}'.format(self.banners_type)
        last_processed_table = self.yt_client.get_attribute(self.monitoring_path, attribute_name, None)
        hour_log_tables = self.yt_client.list(HOUR_LOG_PATH, sort=True)
        hour_log_table = ''
        if last_processed_table:
            for table in hour_log_tables:
                if table > last_processed_table:
                    hour_log_table = table
                    break
        elif hour_log_tables:
            hour_log_table = hour_log_tables[-1]

        return hour_log_table

    def _get_fixed_clients(self):
        return [client['domain_id'] for client in self.required_clients]

    @property
    def _is_need_preprocess_fs(self):
        if self.yt_client.exists(self.preprocessed_full_state):
            attribute_name = 'original_fs'
            old_original_fs = self.yt_client.get(self.preprocessed_full_state, attributes=[attribute_name]).attributes.get(attribute_name, None)
            if old_original_fs == self.fs_id:
                return False
        return True
