# coding: utf-8

import argparse
import itertools
import json
import logging
import md5
import os
import os.path
import random
import six
import six.moves.urllib.parse
import time
import urlpy

import yt.yson as yson
import yt.wrapper as yt

import bm.yt_tools
from bannerland.yql.tools import do_yql, list_yql_result, get_client as get_yql_client
from irt.bannerland.options import get_option as get_bl_opt, get_cypress_config

from bannerland.yt_local import get_proxy as get_local_proxy
from bannerland.make_banners import get_maker_cls
from bannerland.utils import deep_diff


def second_level_domain(url):
    domain = six.moves.urllib.parse.urlparse(url).netloc
    domain = '.'.join(domain.split('.')[-2:])
    return domain


def publish_fs_to_deltas(yt_client, task_type, fs_dir=None, dry_run=False):
    conf = get_cypress_config(task_type)
    if fs_dir is None:
        arc = conf.get_path('full_state_archive')
        fs_dir = max(yt_client.list(arc, absolute=True))

    if not yt_client.get_attribute(fs_dir,  'archive_worker_export_deltas', None):
        raise RuntimeError("Wait for export_deltas worker!")

    timestamp = int(time.time())
    final_dir = yt.ypath_join(fs_dir, 'final')
    dst_dir = yt.ypath_join(conf.get_path('deltas_export'), 'v1')
    table_types = ['banners']
    copy = {yt.ypath_join(final_dir, tt): yt.ypath_join(dst_dir, tt, '{}.fs'.format(timestamp)) for tt in table_types}

    with yt_client.Transaction():
        for src, dst in copy.items():
            logging.warning('will copy {} => {}'.format(src, dst))
            if not dry_run:
                yt_client.copy(src, dst)
                yt_client.set_attribute(dst, 'DeltaTimestamp', timestamp)
                logging.warning('copy done!')


def init_yt_dirs(yt_client, task_type):
    conf = get_cypress_config(task_type)
    root = conf.root
    for name, yt_dir in conf.paths.items():
        if 'dyntables/' in yt_dir:
            # hell spike: this may be a dyntable path, so create only parent dir
            yt_dir = os.path.dirname(yt_dir)
        yt_path = root + '/' + yt_dir
        logging.warning('will create %s:  %s', name, yt_path)
        yt_client.create('map_node', yt_path, recursive=True, ignore_existing=True)


def copy_table(src_client, src_table, dst_client, dst_table, attrs=()):
    if src_client.config['proxy']['url'] == dst_client.config['proxy']['url']:
        dst_client.copy(src_table, dst_table, recursive=True, force=True)
    else:
        schema = src_client.get_attribute(src_table, 'schema')
        dst_client.create('table', path=dst_table, attributes={'schema': schema}, recursive=True, ignore_existing=True)
        dst_client.write_table(dst_table, src_client.read_table(src_table))
    for attr in attrs:
        val = src_client.get_attribute(src_table, attr)
        dst_client.set_attribute(dst_table, attr, val)


def cmd_get_tao(task_type, yt_client, output_table, input_tao_table=None, pocket=None, last_pocket=False, tasks_count=None, min_task_offers=None, max_task_offers=None):
    cypress_conf = get_cypress_config(task_type)
    arc_dir = cypress_conf.get_path('make_banners_archive')
    if input_tao_table is None:
        if pocket is not None:
            pocket = pocket.split('/')[-1]
        elif last_pocket:
            pockets = sorted(yt_client.list(arc_dir), reverse=True)
            for pk in pockets:
                if yt_client.get_attribute(arc_dir + '/' + pk, 'archive_worker_monitor', None):
                    pocket = pk
                    break
        if pocket is None:
            raise Exception("Can't get input table!")
        input_tao_table = arc_dir + '/' + pocket + '/tasks_and_offers'

    if not yt_client.exists(input_tao_table):
        raise Exception("Input tao table {} does not exist".format(input_tao_table))

    logging.warning('will use tao table: %s', input_tao_table)

    if tasks_count is None and min_task_offers is None and max_task_offers is None:
        yt_client.copy(input_tao_table, output_table, force=True, recursive=True)
        return

    yt_cluster = yt_client.config['proxy']['url'].split('.')[0]
    yql_client = get_yql_client(db=yt_cluster)
    query = """
        select task_id, count(*) as offers_count
        from `{input}`
        group by task_id
    """.format(input=input_tao_table)

    tasks_data = list_yql_result(do_yql(yql_client, query))
    if min_task_offers is not None:
        tasks_data = [r for r in tasks_data if r['offers_count'] >= min_task_offers]
    if max_task_offers is not None:
        tasks_data = [r for r in tasks_data if r['offers_count'] <= max_task_offers]

    tasks = [r['task_id'] for r in tasks_data]
    random.shuffle(tasks)
    if tasks_count is not None:
        tasks = tasks[:tasks_count]

    with yt_client.TempTable() as tmp:
        yt_client.write_table(tmp, [{'task_id': tid} for tid in tasks])
        join_query = """
            pragma yt.InferSchema = '1';
            pragma SimpleColumns;

            insert into `{output}`
                with truncate
            select *
            from `{input}` left semi join `{tasks}` using (task_id)
        """.format(input=input_tao_table, output=output_table, tasks=tmp)
        do_yql(yql_client, join_query)


def copy_last_tao(yt_client, task_type, max_rows=None):
    main_conf = get_cypress_config(task_type)

    os.environ['BL_PERF_MODE'] = os.environ['BL_DYN_MODE'] = 'prod'
    prod_conf = get_cypress_config(task_type)

    src_dir = prod_conf.get_path('tao_current')
    dst_dir = main_conf.get_path('tao_current')
    if src_dir == dst_dir:
        raise ValueError("Can't copy: {} = {}".format(src_dir, dst_dir))

    last_tao = max(yt_client.list(src_dir))

    src = yt.ypath_join(src_dir, last_tao)
    dst = yt.ypath_join(dst_dir, last_tao)
    if max_rows is None:
        logging.info('will copy %s => %s', src, dst)
        yt_client.copy(src, dst, force=True)
    else:
        with yt.TempTable() as tmp:
            logging.info('will merge %d rows %s => %s', max_rows, src, dst)
            src = yt.TablePath(src, end_index=max_rows)
            yt_client.run_merge(src, tmp)
            yt_client.move(tmp, dst, force=True)


def copy_last_archive(yt_client, task_type, arc_type, previous=None):
    main_conf = get_cypress_config(task_type)

    os.environ['BL_PERF_MODE'] = os.environ['BL_DYN_MODE'] = 'prod'
    prod_conf = get_cypress_config(task_type)

    if arc_type == 'make_banners':
        arc_name = 'make_banners_archive'
    elif arc_type == 'full_state':
        arc_name = 'full_state_archive'
    else:
        raise ValueError('Bad arc_type: {}'.format(arc_type))

    src_dir = prod_conf.get_path(arc_name)
    dst_dir = main_conf.get_path(arc_name)
    if src_dir == dst_dir:
        raise ValueError("Can't copy: {} = {}".format(src_dir, dst_dir))

    dirs = sorted(yt_client.list(src_dir), reverse=True)
    if previous is None:
        last_dir = dirs[0]
    else:
        last_dir = dirs[previous]

    src_path = yt.ypath_join(src_dir, last_dir)
    dst_path = yt.ypath_join(dst_dir, last_dir)
    logging.info('will copy %s => %s', src_path, dst_path)
    yt_client.copy(src_path, dst_path, recursive=True)


def set_prod_attrs(yt_client, task_type, skip=0):
    attr_name = 'last_avatars_put_table'
    tao_dir = get_cypress_config(task_type).get_path('tao_current')
    tao_list = list(reversed(yt_client.list(tao_dir, sort=True)))
    tao_table = tao_list[skip]
    logging.warning('will set attr %s = %s for %s', attr_name, tao_table, tao_dir)
    yt_client.set_attribute(tao_dir, attr_name, tao_table)


def fetch_external_sources(task_type, src_client, main_client, dst_client, input_tao_table, output_dir):
    yt_cluster = main_client.config['proxy']['url'].split('.')[0]
    yql_client = get_yql_client(db=yt_cluster)
    src_client.create('map_node', output_dir, recursive=True, ignore_existing=True)

    def parse_domain(pt_inf):
        pt = json.loads(pt_inf)
        return second_level_domain(six.ensure_str(urlpy.parse(six.ensure_text(pt['url'])).punycode().unicode))

    def parse_domain_mapper(row):
        yield {'domain': parse_domain(row['product_inf'])}

    domains_table = yt.ypath_join(output_dir, 'domains')
    src_client.run_map_reduce(
        parse_domain_mapper,
        bm.yt_tools.FirstReducer(),
        input_tao_table,
        domains_table,
        reduce_by=['domain'],
    )

    if src_client.row_count(domains_table) <= 100:
        domains = [row['domain'] for row in src_client.read_table(domains_table)]
        domain_filter = """
            where `domain` in ({domain_list})
        """.format(domain_list=",".join(['"{}"'.format(dom) for dom in domains]))
    else:
        copy_table(src_client, domains_table, dst_client, domains_table)
        domain_filter = """
            left semi join `{domains_table}` as dom using(domain)
        """.format(domains_table=domains_table)
    src_client.remove(domains_table)

    ext_srcs = get_maker_cls(task_type).get_external_sources()

    yql_query = """
        pragma yt.ForceInferSchema = '1';
    """
    result = []
    for src_num, ext_src in enumerate(ext_srcs):
        ext_table = ext_src.pop('table')
        output = yt.ypath_join(output_dir, '/fetched_external_{0:02d}_{1}'.format(src_num, ext_table.split('/')[-1]))
        yql_query_part = """
            insert into `{output_table}`
                with truncate
            select *
            from `{input_table}`
            {domain_filter};
        """.format(
            input_table=ext_table,
            output_table=output,
            domain_filter=domain_filter,
        )
        yql_query += yql_query_part
        result.append((output, ext_src))

    do_yql(yql_client, yql_query)
    for table, info in result:
        main_client.set_attribute(table, 'external_source_info', info)
        copy_table(main_client, table, dst_client, table, attrs=['external_source_info'])


def compare_gen_yt(yt_client, task_type, old_table, new_table, diff_table=None, diff_file=None):
    if diff_file is not None:
        diff_fh = open(diff_file, 'w')

    @yt.with_context
    def annotate_rows(row, context):
        if context.table_index == 0:
            row['__src__'] = 'old'
        else:
            row['__src__'] = 'new'
        yield row

    def diff_preprocess(rows):
        # key = id(row), remove dups and sort
        seen_keys = set()
        result = []
        json_fields = ['BSInfo', 'ModelCard', 'BSData']
        out_of_diff_fields = ['row_id', 'UpdateTime', 'BannerlandBeginTime', 'Info', 'bannerphrase_md5', 'product_md5']
        for row in rows:
            for field in out_of_diff_fields:
                row.pop(field, None)

            if task_type == 'dyn':  # TEMP SPIKE
                no_iron_fields = ["MinusScore", "OrderID", "Score", "bl_phrase_template_type"]
                for f in no_iron_fields:
                    row.pop(f, None)
            for k, v in row.items():
                if isinstance(v, str):
                    row[k] = v.decode('utf-8', 'ignore')
            for f in json_fields:
                if f in row and row[f]:
                    row[f] = json.loads(row[f])
            row_key = md5.new(json.dumps(row, sort_keys=True)).hexdigest()
            if row_key in seen_keys:
                continue
            seen_keys.add(row_key)
            row['__sortkey__'] = row_key
            result.append(row)

        result.sort(key=lambda x: x['__sortkey__'])
        for row in result:
            row.pop('__sortkey__')

        return seen_keys, result

    def diff_reduce(key, rows):
        old_rows = []
        new_rows = []
        for row in rows:
            src = row.pop('__src__')
            if src == 'old':
                old_rows.append(row)
            else:
                new_rows.append(row)
        old_keys, old_rows = diff_preprocess(old_rows)
        new_keys, new_rows = diff_preprocess(new_rows)
        res = {
            'old': len(old_rows),
            'new': len(new_rows),
            'gone': len(old_keys - new_keys),
            'came': len(new_keys - old_keys),
            'diff': deep_diff(old_rows, new_rows),
        }
        if res['diff']:
            res['old_rows'] = old_rows
            res['new_rows'] = new_rows
        res.update(key)
        yield res

    tasks_changed = 0
    group_fields = ['task_id', 'Url', 'Text']
    diff_fields = ['old', 'new', 'gone', 'came']
    total_count = {f: 0 for f in diff_fields}
    with yt_client.TempTable() as tmp_full, yt_client.TempTable() as tmp_counts, yt_client.TempTable() as tmp_view:
        diff_schema = [{'name': f, 'type': 'uint64'} for f in diff_fields]
        diff_schema += [{'name': f, 'type': 'string'} for f in group_fields]
        diff_schema += [{'name': f, 'type': 'any'} for f in ['diff', 'old_rows', 'new_rows']]
        yt_client.alter_table(tmp_full, schema=diff_schema)
        yt_client.alter_table(tmp_view, schema=diff_schema)

        yt_client.run_map_reduce(
            annotate_rows,
            diff_reduce,
            [old_table, new_table],
            tmp_full,
            reduce_by=group_fields,
            spec={'reduce_job_io': {'table_writer': {'max_row_weight': 128 * yt.common.MB}}}
        )

        def select_diff(row):
            if row.get('diff'):
                yield row
        yt_client.run_map(
            select_diff,
            tmp_full,
            tmp_view,
            spec={'job_io': {'table_writer': {'max_row_weight': 128 * yt.common.MB}}},
        )
        yt_client.run_sort(tmp_view, sort_by=group_fields)
        if diff_table is not None:
            yt_client.copy(tmp_view, diff_table, force=True)
            logging.info('Created diff table: %s', diff_table)

        # Это можно заменить на YQL!
        def cut_fields(row):
            yield {f: row[f] for f in diff_fields + ['task_id']}

        def sum_fields(key, rows):
            counter = {f: 0 for f in diff_fields}
            for row in rows:
                for f in diff_fields:
                    counter[f] += row.get(f)
            yield {'task_id': key['task_id'], 'counter': counter}

        yt_client.run_map_reduce(
            cut_fields,
            sum_fields,
            tmp_full,
            tmp_counts,
            reduce_by=['task_id'],
        )
        task_count = {row['task_id']: row['counter'] for row in yt_client.read_table(tmp_counts)}

        # ограничиваем кол-во строк, чтобы не создавать слишком большие деревья и огромные логи
        max_diff_per_task = 1000
        for task_id, rows_iter in itertools.groupby(yt_client.read_table(tmp_view), lambda row: row['task_id']):
            if diff_file is not None:
                diff_fh.write('Task {}:\n'.format(task_id))
            tree = {}
            seen_diff = 0
            for row in rows_iter:
                if row['diff']:
                    seen_diff += 1
                    if seen_diff == max_diff_per_task:
                        break
                    node = tree
                    path = [f + '=' + row[f] for f in group_fields]
                    for i in range(len(path)):
                        p = path[i]
                        if p not in node:
                            if i == len(path)-1:
                                node[p] = []
                            else:
                                node[p] = {}
                        node = node[p]
                    node.append({'diff': row['diff'], 'old': row['old_rows'], 'new': row['new_rows']})

            curr_count = task_count.get(task_id, {})
            if curr_count['gone'] or curr_count['came']:
                tasks_changed += 1
                if diff_file is not None:
                    diff_fh.write('Task {} changed!\n'.format(task_id))
                    for f in diff_fields:
                        diff_fh.write(' {}: {}\n'.format(f, curr_count[f]))
                    if seen_diff == max_diff_per_task:
                        diff_fh.write(' diff is too large; it is truncated!\n')
                    diff_fh.write(json.dumps(tree, ensure_ascii=False, indent=2, sort_keys=True) + "\n")

    if diff_file is not None:
        diff_fh.close()

    print '=== Task summary ==='
    for task_id, counts in task_count.items():
        summary = ', '.join(['{}: {}'.format(f, counts[f]) for f in diff_fields])
        if counts['gone'] or counts['came']:
            summary += '; DIFF'
        else:
            summary += '; OK'
        print 'Task {}: {}'.format(task_id, summary)
    print ''

    total_count = {f: 0 for f in diff_fields}
    for counter in task_count.values():
        for f in diff_fields:
            total_count[f] += counter[f]

    print '=== Total ==='
    print 'tasks changed: {} of {}'.format(tasks_changed, len(task_count.keys()))
    for f in diff_fields:
        print '{}: {}'.format(f, total_count[f])


def create_dyntable(name, path=None):
    if name in ['perf_tasks', 'dyn_tasks']:
        fields = [
            {'name': 'task_id',             'type': 'string',   'required': True, 'sort_order': 'ascending'},
            {'name': 'property',            'type': 'string',   'required': True, 'sort_order': 'ascending'},
            {'name': 'value',               'type': 'any',      'required': False},
            {'name': 'pocket',              'type': 'string',   'required': False},
            {'name': 'update_time',         'type': 'string',   'required': False},
        ]
        if path is None:
            task_type_d = {
                'perf_tasks': 'perf',
                'dyn_tasks': 'dyn',
            }
            cypress_config = get_cypress_config(task_type_d[name])
            path = cypress_config.get_path('tasks_info_dyntable')
    elif name == 'perf_avatars_cache':
        fields = [
            {'name': 'url_hash',        'type': 'uint64',   'required': False, 'sort_order': 'ascending', 'expression': 'farm_hash(url)'},
            {'name': 'url',             'type': 'string',   'required': True, 'sort_order': 'ascending'},
            {'name': 'response_code',   'type': 'int32',    'required': True},
            {'name': 'avatars',         'type': 'string',   'required': False},
            {'name': 'meta',            'type': 'any',      'required': False},
            {'name': 'update_time',     'type': 'uint64',   'required': True},
        ]
        if path is None:
            path = get_bl_opt('perf_avatars_cache_dyntable')

    schema = yson.YsonList(fields)
    schema.attributes["strict"] = True

    assert not yt.exists(path)
    logging.info('will create table: %s', path)
    bm.yt_tools.create_dynamic_table(path, schema, 'bannerland')


if __name__ == '__main__':
    logging.basicConfig(level=logging.INFO)
    formatter = argparse.ArgumentDefaultsHelpFormatter
    parser = argparse.ArgumentParser(formatter_class=formatter)
    subparsers = parser.add_subparsers(dest='cmd')

    dyntable_parser = subparsers.add_parser('create_dyntable', formatter_class=formatter)
    dyntable_parser.add_argument('name', choices=[
        'perf_tasks', 'perf_avatars_cache',
        'dyn_tasks',
    ])
    dyntable_parser.add_argument('--path')

    fetch_external_parser = subparsers.add_parser(
        'fetch_external_sources',
        formatter_class=formatter,
        help='get external sources data from big yt for given tasks-and-offers',
    )
    fetch_external_parser.add_argument('--task-type', choices=['dyn', 'perf'], default='perf')
    fetch_external_parser.add_argument('--input-tao-table', help='tasks-and-offers table')
    fetch_external_parser.add_argument('--output-dir', help='dir for external tables')
    fetch_external_parser.add_argument('--local-input', action='store_true', help='input tao table is on local yt')
    fetch_external_parser.add_argument('--local-output', action='store_true', help='write output tables to local yt')

    get_tao_parser = subparsers.add_parser(
        'get_tao',
        formatter_class=formatter,
        help='get tasks-and-offers table',
    )
    get_tao_parser.add_argument('--input-tao-table', help='input table to use')
    get_tao_parser.add_argument('--task-type', choices=['dyn', 'perf'], default='perf')
    get_tao_parser.add_argument('--pocket', help='use given pocket, e.g. "2019-03-24_17:40:23" (or full path)')
    get_tao_parser.add_argument('--last-pocket', action='store_true', help='use last processed pocket')
    get_tao_parser.add_argument('--tasks-count', type=int, help='limit for number of tasks')
    get_tao_parser.add_argument('--max-task-offers', type=int, help='maximum number of offers per task')
    get_tao_parser.add_argument('--min-task-offers', type=int, help='minimum number of offers per task')
    get_tao_parser.add_argument('--mode', choices=['local'], default=None)
    get_tao_parser.add_argument('--output-table', required=True)

    copy_last_tao_parser = subparsers.add_parser(
        'copy_last_tao',
        formatter_class=formatter,
        help='copy last tao table from prod',
    )
    copy_last_tao_parser.add_argument('--task-type', choices=['dyn', 'perf'], default='perf')
    copy_last_tao_parser.add_argument('--max-rows', type=int)

    copy_last_archive_parser = subparsers.add_parser(
        'copy_last_archive',
        formatter_class=formatter,
        help='copy last make_banners/full_state dir from prod',
    )
    copy_last_archive_parser.add_argument('--task-type', choices=['dyn', 'perf'], default='perf')
    copy_last_archive_parser.add_argument('--arc-type', choices=['make_banners', 'full_state'], required=True)
    copy_last_archive_parser.add_argument('--previous', type=int, help='take n-th oldest dir')

    set_prod_attrs_parser = subparsers.add_parser(
        'set_prod_attrs',
        formatter_class=formatter,
        help='set attrs for make_pocket',
    )
    set_prod_attrs_parser.add_argument('--task-type', choices=['dyn', 'perf'], default='perf')
    set_prod_attrs_parser.add_argument('--skip', type=int, default=0, help='skip last N tables')

    copy_table_parser = subparsers.add_parser('copy_table', help='copy table from/to local yt', formatter_class=formatter)
    copy_table_parser.add_argument('--src', required=True, help='source table path')
    copy_table_parser.add_argument('--dst', required=False, help='destination table path')
    copy_table_parser.add_argument(
        '--local', choices=['src', 'dst'], required=True,
        help='use "src" to copy local->main; "dst" to copy main->local',
    )

    compare_parser = subparsers.add_parser('compare_tables', formatter_class=formatter)
    compare_parser.add_argument('old_table')
    compare_parser.add_argument('new_table')
    compare_parser.add_argument('--task-type', choices=['dyn', 'perf'], default='perf')
    compare_parser.add_argument('--diff-table')
    compare_parser.add_argument('--diff-file')
    compare_parser.add_argument('--mode', choices=['local'], default=None)

    init_yt_dirs_parser = subparsers.add_parser('init_yt_dirs', formatter_class=formatter, help='create yt dirs from PerfYTConfig, DynYTConfig')
    init_yt_dirs_parser.add_argument('--mode', choices=['local'], default=None)
    init_yt_dirs_parser.add_argument('--task-type', choices=['dyn', 'perf'], default='perf')

    publish_fs_parser = subparsers.add_parser('publish_fs_to_deltas', formatter_class=formatter)
    publish_fs_parser.add_argument('--task-type', choices=['dyn', 'perf'], default='perf')
    publish_fs_parser.add_argument('--fs-dir', help='if not defined, use last')
    publish_fs_parser.add_argument('--dry-run', action='store_true')

    args = parser.parse_args()

    if args.cmd == 'create_dyntable':
        create_dyntable(args.name, args.path)
    elif args.cmd == 'fetch_external_sources':
        local_client = yt.YtClient(proxy=get_local_proxy())
        main_client = yt
        fetch_external_sources(
            task_type=args.task_type,
            src_client=local_client if args.local_input else main_client,
            main_client=main_client,
            dst_client=local_client if args.local_output else main_client,
            input_tao_table=args.input_tao_table,
            output_dir=args.output_dir,
        )
    elif args.cmd == 'get_tao':
        with yt.TempTable() as tmp:
            kwargs = vars(args)
            kwargs.pop('cmd')
            task_type = kwargs.pop('task_type')
            mode = kwargs.pop('mode')
            output = kwargs.pop('output_table')
            cmd_get_tao(task_type, yt, tmp, **kwargs)
            if mode == 'local':
                local_client = yt.YtClient(proxy=get_local_proxy())
                copy_table(yt, tmp, local_client, output)
            else:
                yt.copy(tmp, output, force=True, recursive=True)
    elif args.cmd == 'copy_last_tao':
        copy_last_tao(yt, args.task_type, max_rows=args.max_rows)
    elif args.cmd == 'copy_last_archive':
        copy_last_archive(yt, args.task_type, args.arc_type, previous=args.previous)
    elif args.cmd == 'set_prod_attrs':
        set_prod_attrs(yt, args.task_type, args.skip)
    elif args.cmd == 'copy_table':
        local_client = yt.YtClient(proxy=get_local_proxy())
        main_client = yt
        if args.local == 'src':
            src_client = local_client
            dst_client = main_client
        else:
            src_client = main_client
            dst_client = local_client
        copy_table(src_client, args.src, dst_client, args.dst or args.src)
    elif args.cmd == 'compare_tables':
        if args.mode == 'local':
            yt_client = yt.YtClient(proxy=get_local_proxy())
        else:
            yt_client = yt
        compare_gen_yt(yt_client, args.task_type, args.old_table, args.new_table, args.diff_table, args.diff_file)
    elif args.cmd == 'init_yt_dirs':
        if args.mode == 'local':
            yt_client = yt.YtClient(proxy=get_local_proxy())
        else:
            yt_client = yt
        init_yt_dirs(yt_client, args.task_type)
    elif args.cmd == 'publish_fs_to_deltas':
        publish_fs_to_deltas(yt, args.task_type, fs_dir=args.fs_dir, dry_run=args.dry_run)
