#!/usr/bin/env python
# coding: utf-8

import json
import re
import getpass
import datetime
import logging
import sys
import os
import os.path
import signal
import argparse
import subprocess
import multiprocessing


logging.basicConfig(level=logging.INFO, format="%(asctime)s\t[%(process)d]\t%(levelname)s\t%(message)s")
logging.getLogger('yql.client.request').setLevel(logging.ERROR)  # suppress useless warnings

DYNSMART_PATH = 'rt-research/broadmatching/scripts/dyn-smart-banners'

#
# Util functions
#

def remove_comments(json_like):
    return re.sub('#.*', '', json_like)

# copy-paste from internet
def remove_trailing_commas(json_like):
    trailing_object_commas_re = re.compile(
        r'(,)\s*}(?=([^"\\]*(\\.|"([^"\\]*\\.)*[^"\\]*"))*[^"]*$)')
    trailing_array_commas_re = re.compile(
        r'(,)\s*\](?=([^"\\]*(\\.|"([^"\\]*\\.)*[^"\\]*"))*[^"]*$)')
    # Fix objects {} first
    objects_fixed = trailing_object_commas_re.sub("}", json_like)
    # Now fix arrays/lists [] and return the result

    return trailing_array_commas_re.sub("]", objects_fixed)


def get_arcadia_root():
    mod_path = os.path.abspath(__file__)
    return os.path.abspath(os.path.dirname(mod_path) + '/../../..')

def set_last_symlink(root, name):
    if not os.path.exists(root):
        os.mkdir(root)
    last_symlink = os.path.join(root, 'last')
    try:
        os.remove(last_symlink)
    except:
        pass
    os.symlink(name, last_symlink)


#
# Основной класс для выполнения команд в рамках одного конфига
#

class BLKitWorker():
    fixtures_yt_table_alias = {
        'perf_big': '//home/bannerland/data/fixtures/perftasks_1k',
        'perf_small': '//home/bannerland/data/fixtures/perftasks_small',
        'dyn_big': '//home/bannerland/data/fixtures/dyntasks_big',
        'dyn_small': '//home/bannerland/data/fixtures/dyntasks_small',
    }

    def __init__(self, arcadia_root, task_type=None, work_dir=None, yt_dir=None, ya_lock=None, ya_cache=None, no_ya_make=False, no_rsync_resources=False, local=False, children_registry=None,
                 relax=False):
        self.arcadia_root = arcadia_root
        self.task_type = task_type
        if work_dir is not None:
            self.work_dir = work_dir
            os.makedirs(work_dir)
        self.yt_dir = yt_dir
        self.ya_lock = ya_lock
        self.ya_cache = ya_cache if ya_cache is not None else {}
        self.no_ya_make = no_ya_make
        self.no_rsync_resources = no_rsync_resources
        self.local = local
        self.relax = relax
        self.children_registry = children_registry if children_registry is not None else {}

    # вызов ya make с учётом:
    # * локов -  при одновременном запуске из разных процессов будет фейл
    # * кеширования -  ускоряемся за счёт своего shared кэша
    # * флага no_ya_make
    def ya_make(self, target, force=False):
        if self.no_ya_make and not force:
            return
        is_cached = False
        if self.ya_cache.get(target):
            is_cached = True
        else:
            if self.ya_lock is not None:
                logging.info('lock for ya make %s ...', self.arcadia_root)
                self.ya_lock.acquire()
                logging.info('lock acquired')
            if self.ya_cache.get(target):
                is_cached = True
            else:
                self.log_call(['./ya', 'make', '-r', '--checkout', target], cwd=self.arcadia_root)
                self.ya_cache[target] = True
            if self.ya_lock is not None:
                self.ya_lock.release()
                logging.info('lock released')
        if is_cached:
            logging.info('ya make %s at %s is already done!', target, self.arcadia_root)

    def log_call(self, cmd, add_env=None, add_root=False, **kwargs):
        call_kwargs = kwargs.copy()
        if add_env is None:
            add_env = {}
        if add_root:
            add_env['ARCADIA_ROOT'] = self.arcadia_root

        env = os.environ.copy()
        env.update(add_env)
        call_kwargs['env'] = env

        logging.info('call cmd: %s, with add_env: %s and kwargs: %s', cmd, add_env, kwargs)
        proc = subprocess.Popen(cmd, **call_kwargs)
        pid = proc.pid
        logging.info('started subprocess with pid: %d', pid)
        self.children_registry[pid] = 1
        exitcode = proc.wait()
        self.children_registry[pid] = 0
        if exitcode != 0:
            logging.error('subprocess failed: %s', cmd)
            raise Exception('subprocess failed')
        logging.info('subprocess %d finished OK', pid)

    def sync_arcadias(self, root1, root2):
        logging.info('will sync arcadias %s vs %s', root1, root2)
        self.log_call(['./ya', 'tool', 'svn', 'update', '.'], cwd=root1)
        svn_info = subprocess.check_output(['./ya', 'tool', 'svn', 'info', '.'], cwd=root1)
        rev_info = re.search('Revision: (\d+)', svn_info)
        rev = rev_info.groups()[0]
        logging.info('arcadia revision: %s', rev)
        self.log_call(['./ya', 'tool', 'svn', 'update', '--revision', str(rev), '.'], cwd=root2)

    def update_rsync_resources(self):
        if self.no_rsync_resources:
            return
        bl_gendicts = ['generated_dicts', 'models', 'marketdata_subphraser', 'db_dump_bannerland']  # TODO унести этот список в конфиг
        if self.task_type == 'dyn':
            bl_gendicts += ['dyn_stat']

        self.log_call([
            'rt-research/broadmatching/scripts/get-resources.pl',
            '--list', ','.join(bl_gendicts),
        ], cwd=self.arcadia_root)

    def run_command(self, cmd, opts):
        name2func = {
            'renew_local_yt': self.cmd_renew_local_yt,
            'get_tasks': self.cmd_get_tasks,
            'preview_offers': self.cmd_preview_offers,
            'export_offers': self.cmd_export_offers,
            'upload_offers': self.cmd_upload_offers,
            'make_banners': self.cmd_make_banners,
            'generate_native': self.cmd_generate_native,
            'compare_tables': self.cmd_compare_tables,
        }
        name2func[cmd](opts)

    def cmd_renew_local_yt(self, opts):
        root = self.arcadia_root
        target = 'rt-research/bannerland/tools/yt_local'
        binary = target + '/' + target.split('/')[-1]
        self.ya_make(target)
        try:
            self.log_call([binary, 'stop'], cwd=root)
        except:
            logging.info('local yt is not running')
        try:
            self.log_call([binary, 'delete'], cwd=root)
        except:
            logging.info('some problem with yt deletion')

        download_args = [binary, 'download', '--stable']
        self.log_call(download_args, cwd=root)

        self.log_call([binary, 'start'], cwd=root)

    def cmd_get_tasks(self, opts):
        self.ya_make('rt-research/broadmatching')
        binary = os.path.join(self.arcadia_root, DYNSMART_PATH, 'get-taskjson.pl')
        task_dir = self.work_dir + '/tasks'
        if not os.path.exists(task_dir):
            os.mkdir(task_dir)
        if opts.get('get_tasks'):
            add_options = ['--' + kv for kv in opts['get_tasks']]
        else:
            add_options = []

        if self.relax:
            add_options.append('--relax')

        task_ids = []
        task_table = opts.get('task_table')

        if opts.get('task_id'):
            task_ids += opts['task_id']

        if not task_ids and not task_table:
            # не задали ничего -> запускаемся по конфигу
            with open(opts['task_conf']) as conf_fh:
                conf_str = conf_fh.read()
                conf_str = remove_comments(conf_str)
                conf_str = remove_trailing_commas(conf_str)
                conf = json.loads(conf_str)
                data = conf.get(self.task_type, {})
                task_ids += data.get('task_ids', [])
                task_table = data.get('tasks_yt_table')

        main_args = [binary, '--task-type', self.task_type]
        add_args = []
        add_env = {}
        if task_table:
            # Отключаем хождение во внешний интернет для фикстур
            if task_table in self.fixtures_yt_table_alias or task_table in self.fixtures_yt_table_alias.values():
                add_env['BM_DISABLE_EXTERNAL_INTERNET'] = '1'
            task_table = self.fixtures_yt_table_alias.get(task_table, task_table)
            add_args += ['--task-table', task_table]

        if task_ids:
            # отдельные таски из task_table  или продовой если не задано
            for task_id in task_ids:
                add_args += ['--task-id', task_id]
                add_args += ['--output-file', task_dir + '/' + task_id + '.json']
        else:
            # запускаемся по всем таскам из task-table
            add_args += ['--output-dir', task_dir]
        self.log_call(main_args + add_args + add_options, add_env=add_env)

    def cmd_preview_offers(self, opts):
        self.ya_make('rt-research/broadmatching')
        self.ya_make('rt-research/bannerland/bin/avatars')
        self.update_rsync_resources()

        binary = os.path.join(self.arcadia_root, DYNSMART_PATH, 'preview-offers.pl')
        output_dir = self.work_dir + '/previews'
        if not os.path.exists(output_dir):
            os.mkdir(output_dir)

        task_file_args = []
        if opts.get('task_file'):
            input_files = opts['task_file']
            for fn in input_files:
                task_file_args += ['--task-file', fn]
        elif opts.get('task_dir'):
            task_file_args = ['--task-dir', opts['task_dir']]
        else:
            task_dir = os.path.join(self.work_dir, 'tasks')
            task_file_args = ['--task-dir', task_dir]

        cmd_args = [binary] + task_file_args + ['--output_dir', output_dir]
        self.log_call(cmd_args)

    def cmd_export_offers(self, opts):
        self.ya_make('rt-research/broadmatching')
        self.update_rsync_resources()

        binary = os.path.join(self.arcadia_root, DYNSMART_PATH, 'export-offers.pl')
        offers_dir = self.work_dir + '/offers'
        if not os.path.exists(offers_dir):
            os.mkdir(offers_dir)

        task_file_args = []
        if opts.get('task_file'):
            input_files = opts['task_file']
            for fn in input_files:
                task_file_args += ['--task-file', fn]
        elif opts.get('task_dir'):
            task_file_args = ['--task-dir', opts['task_dir']]
        else:
            task_dir = os.path.join(self.work_dir, 'tasks')
            task_file_args = ['--task-dir', task_dir]

        cmd_args = [binary, '--task-type', self.task_type] + task_file_args + ['--output_dir', offers_dir]
        if self.relax:
            cmd_args.append('--relax')
        add_env = {
            'BANNERLAND_FORCE_MAPPING': '1',
        }

        if opts.get('export_offers'):
            cmd_args += ['--' + kv for kv in opts['export_offers']]

        # TODO: удалить через некоторое время, скажем, осенью 2019-го (когда все обновят export-offers.pl)
        if self.task_type == 'dyn':
            add_env['BANNERLAND_NO_IRON_GEN'] = '1'
        self.log_call(cmd_args, add_env=add_env)

    def cmd_upload_offers(self, opts):
        bin_target = 'rt-research/bannerland/tools/upload_export_offers'
        bin_script = os.path.join(self.arcadia_root, bin_target, bin_target.split('/')[-1])
        self.ya_make(bin_target)

        file_args = []
        if opts.get('export_offers_file'):
            input_files = opts['export_offers_file']
            for fn in input_files:
                file_args += ['--eo-file', fn]
        elif opts.get('export_offers_dir'):
            file_args = ['--eo-dir', opts.get('export_offers_dir')]
        else:
            offers_dir = os.path.join(self.work_dir, 'offers')
            file_args = ['--eo-dir', offers_dir]

        call_args = [bin_script, '--task-type', self.task_type] + file_args + ['--yt-path', self.yt_dir + '/tasks_and_offers']
        if self.local:
            call_args += ['--mode', 'local']

        self.log_call(call_args, add_root=True)

    def cmd_make_banners(self, opts):
        bin_target = 'rt-research/bannerland/make_banners'
        bin_script = os.path.join(self.arcadia_root, bin_target, bin_target.split('/')[-1])
        self.ya_make(bin_target)

        if opts.get('input_tao_table'):
            input_tao_table = opts['input_tao_table']
        else:
            input_tao_table = self.yt_dir + '/tasks_and_offers'

        if opts['catalogia'] == 'local':
            self.ya_make('rt-research/broadmatching/ya_packages/bm_dicts', force=True)
            self.ya_make('rt-research/broadmatching/ya_packages/bm_bannerland_lib', force=True)

        call_args = [
            bin_script,
            '--task-type', self.task_type,
            '--input-tao-table', input_tao_table,
            '--catalogia', opts['catalogia'],
            '--ensure-external',
        ]

        make_banners_opts = opts.get('make_banners') or []

        if not any(x.startswith('yt-dir=') for x in make_banners_opts):
            # yt-dir не задан явной опцией для бинаря make_banners, берём стандартный
            call_args += ['--yt-dir', self.yt_dir + '/make_banners']

        if opts['fetch_external'] and self.local:
            tool_target = 'rt-research/bannerland/tools'
            tool_binary = tool_target + '/' + tool_target.split('/')[-1]
            self.ya_make(tool_target)

            ext_dir = self.yt_dir + '/external_sources'
            self.log_call([
                tool_binary,
                'fetch_external_sources',
                '--task-type', self.task_type,
                '--input-tao-table', input_tao_table,
                '--output-dir', ext_dir,
                '--local-input',
                '--local-output',
            ], cwd=self.arcadia_root)
            call_args += ['--external-dir', ext_dir]

        if self.local:
            call_args += [
                '--mode', 'local',
                '--fake-archives',
            ]
        call_args += ['--' + kv for kv in make_banners_opts]

        if not opts['add_avatars']:
            call_args += ['--skip-add-avatars']

        # main call to make_banners:
        self.log_call(call_args, cwd=self.arcadia_root)

        if opts['copy_logs'] and self.local:
            tool_target = 'rt-research/bannerland/tools'
            tool_binary = tool_target + '/' + tool_target.split('/')[-1]
            self.ya_make(tool_target)

            log_table = self.yt_dir + '/make_banners/log_merged'
            self.log_call([
                tool_binary,
                'copy_table',
                '--local', 'src',
                '--src', log_table,
                '--dst', log_table,
            ], cwd=self.arcadia_root)
            logging.info('log table copied to big yt: %s', log_table)

    def cmd_get_tao(self, opts):
        target = 'rt-research/bannerland/tools'
        binary = target + '/' + target.split('/')[-1]
        self.ya_make(target)

        call_args = [
            binary, 'get_tao',
            '--task-type', (self.task_type or opts['task_type']),
            '--last-pocket',
            '--output-table', opts['output_table'],
        ]
        for opt in ['tasks_count', 'min_task_offers', 'max_task_offers']:
            if opts.get(opt):
                call_args += ['--' + opt.replace('_', '-'), str(opts[opt])]
        if self.local:
            call_args += ['--mode', 'local']

        add_env = {'BL_PERF_MODE': 'prod', 'BL_DYN_MODE': 'prod'}  # we want to read production tasks-and-offers

        self.log_call(call_args, cwd=self.arcadia_root, add_env=add_env)

    def cmd_generate_native(self, opts):
        self.ya_make('rt-research/broadmatching')
        self.update_rsync_resources()

        call_args = [
            self.arcadia_root + '/rt-research/broadmatching/scripts/dyn-smart-banners/run_native_generation.pl',
            '--save-tao',
            '--task-type', self.task_type,
        ]
        for opt in ['eo_file', 'tao_file', 'tao_table']:
            if opts.get(opt):
                call_args += ['--' + opt, opts[opt]]
                break

        self.log_call(call_args)

    def cmd_compare_tables(self, opts):
        target = 'rt-research/bannerland/tools'
        binary = target + '/' + target.split('/')[-1]
        self.ya_make(target)

        call_args = [
            binary,
            'compare_tables',
            opts['old_table'],
            opts['new_table'],
            '--task-type', (self.task_type or opts['task_type']),
            '--diff-table', opts['diff_table'],
        ]
        if self.local:
            call_args += ['--mode', 'local']
        if opts.get('diff_file'):
            call_args += ['--diff-file', os.path.abspath(opts['diff_file'])]  # abspath нужен из-за смены cwd
        self.log_call(call_args, cwd=self.arcadia_root)


def run_worker(worker, commands, opts):
    for cmd_name in commands:
        logging.info('start cmd: %s', cmd_name)
        worker.run_command(cmd_name, opts.copy())
        logging.info('done cmd: %s', cmd_name)


def main():
    # personal defaults
    os.environ.setdefault('YT_PROXY', 'hahn')
    os.environ.setdefault('YQL_YT_TOKEN_NAME', 'default_yt')
    os.environ.setdefault('YQL_TOKEN_PATH', os.path.expanduser('~/.yql/token'))
    if ('YQL_TOKEN' not in os.environ) and (not os.path.exists(os.environ['YQL_TOKEN_PATH'])):
        logging.warning("Can't get YQL token, put it in ~/.yql/token")

    commands_chain = [
        'get_tasks',
        'export_offers',
        'upload_offers',
        'make_banners',
    ]

    formatter = argparse.ArgumentDefaultsHelpFormatter
    common_parser = argparse.ArgumentParser(add_help=False)

    common_parser.add_argument('--perf', action='store_true', help='perf banners')
    common_parser.add_argument('--dyn', action='store_true', help='dyn banners')
    common_parser.add_argument('--task-type', action='append', choices=['dyn', 'perf'], help='old-fashioned version of --dyn/--perf')

    common_parser.add_argument('--fast', action='store_true', help='do not make any preparations (implies --no-ya-make, --no-rsync-resources etc)')
    common_parser.add_argument('--relax', action='store_true', help='not die if some errors in get-taskjson or export-offers steps')
    common_parser.add_argument('--no-ya-make', action='store_true', help='do not run ya make for used binary targets (but build bm packages anyway)')
    common_parser.add_argument('--no-rsync-resources', action='store_true', help='do not fetch rsync resources')
    common_parser.add_argument('--local', action='store_true', help='use local YT for every command, also use server cdicts in generation, etc')
    common_parser.add_argument('--mode', choices=['local'], default=None, help='alias for --local, deprecated')

    parser = argparse.ArgumentParser(
        parents=[common_parser],
        formatter_class=formatter,
        description='BannerLand Tool Kit, runs various bannerland related commands',
        epilog='Some more examples at:  https://wiki.yandex-team.ru/BM/bl-kit/',
    )
    subparsers = parser.add_subparsers(dest='cmd', title='commands')


    renew_local_yt_parser = subparsers.add_parser('renew_local_yt', formatter_class=formatter, help='get stable local yt sb res')

    def add_get_tasks_args(subparser):
        subparser.add_argument(
            '--task-id', action='append',
            help='task id without chunk, e.g. "2225423_804343436"',
        )
        subparser.add_argument(
            '--task-conf',
            help='tasks config: json-encoded hash {"dyn": {"task_ids": ...}}; used if no ids provided via --task-id',
            default=get_arcadia_root() + '/rt-research/broadmatching/scripts/dyn-smart-banners/b2b/test.conf',
        )

        aliases_str = ', '. join(BLKitWorker.fixtures_yt_table_alias.keys())
        subparser.add_argument(
            '--task-table',
            help='yt table with tasks (now tested only perf); support aliases: "{}"'.format(aliases_str),
        )

        subparser.add_argument('--get-tasks', action='append', help='other options for get-tasks')

    get_tasks_parser = subparsers.add_parser('get_tasks', formatter_class=formatter, help='get task json files from task ids of config')
    add_get_tasks_args(get_tasks_parser)

    def add_preview_offers_args(subparser):
        subparser.add_argument(
            '--task-file', action='append',
            help='task json file',
        )
        subparser.add_argument(
            '--task-dir',
            help='dir with taskjson files',
        )

    preview_offers_parser = subparsers.add_parser('preview_offers', formatter_class=formatter, help='get offer previews (yml2directinf)')
    add_preview_offers_args(preview_offers_parser)

    def add_export_offers_args(subparser):
        subparser.add_argument(
            '--task-file', action='append',
            help='task json file',
        )
        subparser.add_argument(
            '--task-dir',
            help='dir with taskjson files',
        )
        subparser.add_argument('--export-offers', action='append', help='other options for export_offers')

    export_offers_parser = subparsers.add_parser('export_offers', formatter_class=formatter, help='create export_offers file from task')
    add_export_offers_args(export_offers_parser)

    def add_upload_offers_args(subparser):
        subparser.add_argument(
            '--export-offers-file', '--eo-file', action='append',
            help='export_offers tskv file',
        )
        subparser.add_argument(
            '--export-offers-dir', '--eo-dir',
            help='dir with export_offers tskv files',
        )

    upload_offers_parser = subparsers.add_parser('upload_offers', formatter_class=formatter, help='upload export_offers file to YT')
    add_upload_offers_args(upload_offers_parser)

    def add_make_banners_input(subparser):
        subparser.add_argument('--input-tao-table')

    def add_make_banners_opts(subparser):
        subparser.add_argument('--catalogia', default='local', choices=['local', 'sb'], help='source for bm libs and dicts')
        subparser.add_argument('--make-banners', action='append', help='other options for make_banners binary, e.g. --make-banners mode=local')
        subparser.add_argument('--fetch-external', action='store_true', help='get external data from big yt (for mode=local only!)')
        subparser.add_argument('--copy-logs', action='store_true', help='copy logs to big yt (for mode=local only!)')
        subparser.add_argument('--add-avatars', action='store_true', help='Turns add_avatars step on')

    make_banners_parser = subparsers.add_parser('make_banners', formatter_class=formatter, help='run make_banners binary')
    add_make_banners_input(make_banners_parser)
    add_make_banners_opts(make_banners_parser)

    chain_parser = subparsers.add_parser('chain', formatter_class=formatter, help='run chain of commands')
    chain_parser.add_argument('--start', '--from', choices=commands_chain, default=commands_chain[0], help='first command')
    chain_parser.add_argument('--stop', '--to', choices=commands_chain, default=commands_chain[-1], help='last command')
    add_get_tasks_args(chain_parser)
    add_export_offers_args(chain_parser)
    add_upload_offers_args(chain_parser)
    add_make_banners_input(chain_parser)
    add_make_banners_opts(chain_parser)

    def add_get_tao_args(subparser):
        subparser.add_argument('--last-pocket', action='store_true', help='get input from last processed pocket')
        subparser.add_argument('--tasks-count', type=int, help='limit for number of tasks (implies --last-pocket)')
        subparser.add_argument('--min-task-offers', type=int, help='minimum number of offers per task (implies --last-pocket)')
        subparser.add_argument('--max-task-offers', type=int, help='maximum number of offers per task (implies --last-pocket)')

    b2b_parser = subparsers.add_parser('b2b', formatter_class=formatter, help='compare with clean arcadia')
    b2b_parser.add_argument('--start', '--from', choices=commands_chain, default=commands_chain[0], help='command to start with')
    b2b_parser.add_argument('--stop', '--to', choices=commands_chain, default=commands_chain[-1], help='last command')
    b2b_parser.add_argument('--sync', action='store_true', help='use svn up to sync arcadias revision')
    b2b_parser.add_argument('--arcadia-ref', default=os.path.expanduser('~/arcadia_ref'), help='clean arcadia to compare with')
    add_get_tasks_args(b2b_parser)
    add_export_offers_args(b2b_parser)
    add_upload_offers_args(b2b_parser)
    add_make_banners_input(b2b_parser)
    add_make_banners_opts(b2b_parser)
    add_get_tao_args(b2b_parser)

    native_parser = subparsers.add_parser('generate_native', formatter_class=formatter, help='run pure perl script to get native banners')
    native_parser.add_argument('--eo-file', help='export_offers file')
    native_parser.add_argument('--tao-file', help='tasks_and_offers file')
    native_parser.add_argument('--tao-table', help='table with tasks_and_offers, i.e. input-table for make_banners')

    compare_parser = subparsers.add_parser('compare_tables', formatter_class=formatter, help='compare two generated_banners tables')
    compare_parser.add_argument('--old-table', required=True)
    compare_parser.add_argument('--new-table', required=True)
    compare_parser.add_argument('--diff-table', required=True)
    compare_parser.add_argument('--diff-file')

    common_args, unparsed = common_parser.parse_known_args()
    args = parser.parse_args(unparsed)
    opts = vars(args)
    for k in vars(common_args):
        opts.pop(k, None)

    #
    # process parsed args
    #

    if args.cmd in ['renew_local_yt']:
        # команды, для которых не важен task_type, запускаются через base_worker
        task_types = ['any']
    elif common_args.task_type:
        task_types = common_args.task_type
    elif common_args.perf or common_args.dyn:
        task_types = []
        if common_args.perf:
            task_types.append('perf')
        if common_args.dyn:
            task_types.append('dyn')
    else:
        task_types = ['perf']

    logging.info('task types: %s', task_types)

    if common_args.fast:
        common_args.no_ya_make = True
        common_args.no_rsync_resources = True

    if common_args.mode == 'local':
        logging.error('Usage of --mode option is deprecated, write --local instead')
        common_args.local = True

    #
    # prepare configs
    #

    blkit_dir = 'run-bl-kit'
    base_arcadia_root = get_arcadia_root()
    start_timestamp = datetime.datetime.now().strftime('%Y%m%d_%H%M%S')
    base_work_dir = os.path.join(blkit_dir, start_timestamp)
    set_last_symlink(blkit_dir, start_timestamp)
    base_yt_dir = '//home/bannerland/users/{}/{}'.format(getpass.getuser(), base_work_dir)
    base_conf = {
        'arcadia_root': base_arcadia_root,
        'work_dir': base_work_dir,
        'yt_dir': base_yt_dir,
    }
    base_worker = BLKitWorker(base_arcadia_root, local=common_args.local)
    logging.info('bl-kit base config:\n%s', json.dumps(base_conf, indent=4))
    if len(task_types) == 1:
        base_conf['task_type'] = task_types[0]
        configs = [base_conf]
    else:
        configs = []
        for task_type in task_types:
            work_conf = base_conf.copy()
            work_conf['task_type'] = task_type
            work_conf['work_dir'] += '/' + task_type
            work_conf['yt_dir'] += '/' + task_type
            configs.append(work_conf)

    cmd_num = {commands_chain[i]: i for i in range(len(commands_chain))}
    cmds = []
    configs_to_compare = []
    if args.cmd == 'renew_local_yt':
        logging.info('Will start new local yt')
        base_worker.cmd_renew_local_yt(opts)
        cmds = []
    elif args.cmd == 'chain':
        start_cmd = args.start
        stop_cmd = args.stop
        for cmd_name in commands_chain:
            if cmd_num[cmd_name] >= cmd_num[start_cmd] and cmd_num[cmd_name] <= cmd_num[stop_cmd]:
                cmds.append(cmd_name)
    elif args.cmd == 'b2b':
        start_cmd = args.start
        stop_cmd = args.stop
        for cmd_name in commands_chain:
            if cmd_num[cmd_name] >= cmd_num[start_cmd] and cmd_num[cmd_name] <= cmd_num[stop_cmd]:
                cmds.append(cmd_name)

        if opts.get('input_tao_table') and len(task_types) > 1:
            raise Exception("Use --input-tao-table with one task type!")

        if args.tasks_count is not None or args.min_task_offers is not None or args.max_task_offers is not None:
            args.last_pocket = True

        if args.last_pocket:
            if start_cmd != 'make_banners':
                raise Exception("Use --last-pocket, --tasks-count, --max-task-offers, --min-task-offers options with --start make_banners")
            if len(task_types) > 1:
                raise Exception("Specify one task_type!")

        ref_configs = []
        for work_conf in configs:
            parent_yt_dir = work_conf['yt_dir']
            parent_work_dir = work_conf['work_dir']
            ref_conf = work_conf.copy()
            ref_conf['arcadia_root'] = args.arcadia_ref
            work_conf['work_dir'] += '/new'
            work_conf['yt_dir'] += '/new'
            ref_conf['work_dir'] += '/old'
            ref_conf['yt_dir'] += '/old'

            work_conf['parent_yt_dir'] = parent_yt_dir
            ref_conf['parent_yt_dir'] = parent_yt_dir
            work_conf['parent_work_dir'] = parent_work_dir
            ref_conf['parent_work_dir'] = parent_work_dir

            ref_configs.append(ref_conf)

            if args.stop == commands_chain[-1]:
                configs_to_compare.append((ref_conf, work_conf))

            conf_opts = opts.copy()
            if args.last_pocket:
                get_tao_opts = opts.copy()
                tao_table = parent_yt_dir + '/tasks_and_offers.from_last_pocket'
                get_tao_opts['output_table'] = tao_table
                get_tao_opts['task_type'] = task_types[0]
                base_worker.cmd_get_tao(get_tao_opts)
                conf_opts['input_tao_table'] = tao_table

            work_conf['opts'] = conf_opts
            ref_conf['opts'] = conf_opts

        configs += ref_configs

        if args.sync:
            base_worker.sync_arcadias(base_arcadia_root, args.arcadia_ref)
    else:
        cmds.append(args.cmd)

    logging.info('commands to do: %s', cmds)

    #
    # prepare BLKit workers for each config and run in parallel
    #

    ipc_manager = multiprocessing.Manager()
    children_registry = ipc_manager.dict()
    procs = {}
    ya_locks = {}
    ya_caches = {}
    for conf in configs:
        root = conf['arcadia_root']
        if root not in ya_locks:
            ya_locks[root] = multiprocessing.Lock()
            ya_caches[root] = ipc_manager.dict()
        worker = BLKitWorker(
            arcadia_root=root,
            task_type=conf['task_type'],
            work_dir=conf['work_dir'],
            yt_dir=conf['yt_dir'],
            ya_lock=ya_locks[root],
            ya_cache=ya_caches[root],
            no_ya_make=common_args.no_ya_make,
            no_rsync_resources=common_args.no_rsync_resources,
            local=common_args.local,
            relax=common_args.relax,
            children_registry=children_registry,
        )
        proc = multiprocessing.Process(
            target=run_worker,
            args=(worker, conf.get('cmds', cmds), conf.get('opts', opts)),
        )
        proc.start()
        procs[proc.pid] = proc
        logging.info('run_worker [%s] started with pid %d', worker.work_dir, proc.pid)

    logging.info('waiting for sub-processes ...')

    #
    # wait for all workers to finish
    #

    active_pids = set(procs.keys())
    is_failed = False
    while active_pids and not is_failed:
        for pid in list(active_pids):
            proc = procs[pid]
            proc.join(timeout=10)
            if proc.exitcode is None:
                # not finished yet
                continue
            active_pids.remove(pid)
            logging.info("process %d finished with exit code %d", pid, proc.exitcode)
            if proc.exitcode != 0:
                logging.error("process %d failed", pid)
                is_failed = True
                break

    if is_failed:
        for pid in active_pids:
            proc = procs[pid]
            logging.error("will terminate process %d", pid)
            proc.terminate()
        for pid, is_active in children_registry.items():
            if is_active:
                logging.error("will terminate subprocess %d", pid)
                try:
                    os.kill(pid, signal.SIGTERM)
                except:
                    pass

        logging.error('bl-kit failed!')
        return -1

    if configs_to_compare:
        for old_conf, new_conf in configs_to_compare:
            old_table = old_conf['yt_dir'] + '/make_banners/generated_banners.final'
            new_table = new_conf['yt_dir'] + '/make_banners/generated_banners.final'
            diff_table = old_conf['parent_yt_dir'] + '/generated_banners.diff'
            diff_file = old_conf['parent_work_dir'] + '/generated_banners.diff'
            logging.info('Compare %s vs %s', old_table, new_table)
            base_worker.cmd_compare_tables(
                opts={
                    'old_table': old_table,
                    'new_table': new_table,
                    'diff_table': diff_table,
                    'diff_file': diff_file,
                    'task_type': old_conf['task_type'],
                },
            )
            logging.info('See diff in table: %s and file: %s', diff_table, diff_file)

    logging.info('bl-kit: all done!')
    return 0

if __name__ == '__main__':
    exitcode = main()
    sys.exit(exitcode)
