# -*- coding: utf8 -*-

import os.path
import sys
import datetime
import json
import argparse

import yt.wrapper as yt

from irt.bannerland.options import get_cypress_config
from irt.bannerland.options.cypress import PerfYTConfig, DynYTConfig


def get_timestamp():
    return datetime.datetime.now().strftime('%Y-%m-%d %H:%M:%S')  # TODO -> lib


def get_info(task_type, tid):
    dyntable = get_cypress_config(task_type).get_path('tasks_info_dyntable')
    query = "* from [{input_table}] where task_id = '{tid}'".format(input_table=dyntable, tid=tid)
    info = {row.pop('property'): row for row in yt.select_rows(query)}
    return info


def get_last_pocket_name(task_type, tid):
    info = get_info(task_type, tid)
    last_field = 'funnel_str'
    if last_field in info and 'pocket' in info[last_field]:
        return info[last_field]['pocket']
    else:
        raise Exception("No pockets with tid "+tid)


def print_history(task_type, tid):
    archive_dir = get_cypress_config(task_type).get_path('make_banners_archive')
    pockets = yt.list(archive_dir, absolute=True)
    pockets.sort(reverse=True)

    def analyze_pocket_dir(pocket_dir):
        task_table = pocket_dir + '/tasks.final'
        if not yt.is_sorted(task_table):
            return 'pocket not indexed, please wait'

        task_view = yt.TablePath(task_table, exact_key=tid)
        task_rows = list(yt.read_table(task_view))
        if not task_rows:
            return 'task not found!'

        row = task_rows[0]
        return {'bannerphrases_count': row['banners_count']}

    try:
        for pocket_dir in pockets:
            pocket = pocket_dir.split('/')[-1]
            print 'pocket', pocket, ' => ', analyze_pocket_dir(pocket_dir)
    except KeyboardInterrupt:
        print "Interrupted"


def print_from_pocket(task_type, tid, pocket, what):
    tname = {
        'log': 'log_merged',
        'offers': 'tasks_and_offers.final',
        'banners': 'generated_banners.final',
    }
    archive_dir = get_cypress_config(task_type).get_path('make_banners_archive')
    table = os.path.join(archive_dir, pocket, tname[what])
    if not yt.exists(table):
        raise Exception('can\'t find table {} in pocket, maybe it is too old'.format(tname[what]))
    if not yt.is_sorted(table):
        raise Exception('pocket not indexed, please wait')
    table_view = yt.TablePath(table, exact_key=tid)
    for row in yt.read_table(table_view):
        res = []
        for k in sorted(row.keys()):
            if isinstance(row[k], dict):
                row[k] = json.dumps(row[k])
            res.append("{}={}".format(k, row[k]))
        sys.stdout.write("\t".join(res) + "\n")


if __name__ == "__main__":
    yt.config['proxy']['url'] = 'hahn'
    PerfYTConfig.mode = 'prod'  # read-only tool; we need to read prod tables
    DynYTConfig.mode = 'prod'

    common_parser = argparse.ArgumentParser(add_help=False)
    common_parser.add_argument('--task-type', choices=['dyn', 'perf'], default='perf')

    formatter = argparse.ArgumentDefaultsHelpFormatter
    parser = argparse.ArgumentParser(
        parents=[common_parser],
        formatter_class=formatter,
        description='BannerLand Pocket Tool: inspect pockets archive',
    )

    subparsers = parser.add_subparsers(dest='cmd')

    def add_task_arg(subparser):
        subparser.add_argument('task_id')

    def add_pocket_arg(subparser):
        subparser.add_argument('pocket', help='pocket name, e.g.: "2018-12-19_09:07:43"', nargs='?', default='last')

    info_parser = subparsers.add_parser(
        'info', formatter_class=formatter,
        help='information about last task\'s pocket',
    )
    add_task_arg(info_parser)

    hist_parser = subparsers.add_parser(
        'history', formatter_class=formatter,
        help='task in pockets history',
    )
    add_task_arg(hist_parser)

    log_parser = subparsers.add_parser(
        'log', formatter_class=formatter,
        help='generation log, print to stdout',
    )
    add_task_arg(log_parser)
    add_pocket_arg(log_parser)

    offers_parser = subparsers.add_parser(
        'offers', formatter_class=formatter,
        help='input data - exported offers, print to stdout',
    )
    add_task_arg(offers_parser)
    add_pocket_arg(offers_parser)

    banners_parser = subparsers.add_parser(
        'banners', formatter_class=formatter,
        help='generation data, print to stdout',
    )
    add_task_arg(banners_parser)
    add_pocket_arg(banners_parser)

    args = parser.parse_args()
    task_type = args.task_type
    tid = args.task_id
    cmd = args.cmd
    if cmd == 'info':
        info = get_info(task_type, tid)
        print(json.dumps(info, indent=4))
    elif cmd == 'history':
        print_history(task_type, tid)
    elif cmd in {'log', 'offers', 'banners'}:
        pocket = args.pocket
        if pocket == 'last':
            pocket = get_last_pocket_name(task_type, tid)
        print_from_pocket(task_type, tid, pocket, cmd)
