import argparse
import logging
import time

import ads.libs.yql
import yt.wrapper as yt

import irt.bannerland.options


def main(task_type, del_table, yt_client=yt, dry_run=False, output_dir=None):
    conf = irt.bannerland.options.get_cypress_config(task_type)
    fs_archive = conf.get_path('full_state_archive')
    for fs_dir in reversed(yt_client.list(fs_archive, absolute=True, sort=True)):
        fs_table = yt.ypath_join(fs_dir, 'final', 'banners')
        if yt_client.exists(fs_table):
            logging.info('fs_table: %s', fs_table)
            break
        else:
            logging.info('skip fs_dir %s, no final banners table', fs_dir)

    if output_dir is None:
        output_dir = yt.ypath_join(conf.get_path('deltas_export'), 'v1', 'banners')
    logging.info('output_dir: %s', output_dir)

    yt_cluster = yt_client.config['proxy']['url'].split('.')[0]

    with yt_client.TempTable() as tmp_table:
        yql_query = """
            INSERT INTO `{output}` WITH TRUNCATE
            SELECT
                Cast(t_del.BannerID as Int64) as BannerID, true as Deleted
            FROM `{del_table}` as t_del
            LEFT ONLY JOIN `{fs_table}` as t_fs USING(BannerID)
        """.format(fs_table=fs_table, del_table=del_table, output=tmp_table)
        ads.libs.yql.run_yql_query(db=yt_cluster, query=yql_query)

        logging.info('del banners not in fs: %d of %d', yt_client.row_count(tmp_table), yt_client.row_count(del_table))

        timestamp = int(time.time())
        yt_client.set_attribute(tmp_table, 'DeltaTimestamp', timestamp)
        output_table = yt.ypath_join(output_dir, '{}.del_tools'.format(timestamp))

        last_fs_dir = max(yt_client.list(fs_archive, absolute=True))
        if last_fs_dir != fs_dir:
            logging.warning('New fs appeared, check that deltas are not exported!')
            if yt_client.get_attribute(last_fs_dir, 'archive_worker_export_deltas', None):
                raise RuntimeError('export_deltas done for new fs!')

        logging.warning('will copy {} => {}'.format(tmp_table, output_table))
        if dry_run:
            logging.info('dry run, do not copy')
        else:
            yt_client.copy(tmp_table, output_table)
            logging.info('copy done!')


if __name__ == '__main__':
    logging.basicConfig(level=logging.INFO)
    formatter = argparse.ArgumentDefaultsHelpFormatter
    parser = argparse.ArgumentParser(formatter_class=formatter)

    parser.add_argument('del_table')
    parser.add_argument('--task-type', choices=['dyn', 'perf'], default='perf')
    parser.add_argument('--dry-run', action='store_true', help='do not copy to output')
    parser.add_argument('--output-dir', help='change deltas output dir (for test)')

    args = parser.parse_args()

    main(args.task_type, args.del_table, dry_run=args.dry_run, output_dir=args.output_dir)
