# encoding: utf-8

import logging
import os.path
import argparse
import datetime
import random

import yt.wrapper as yt

from bannerland.prepare_input import TasksOffersJoiner, TasksOffersUploader
import bannerland.yt_local

import irt.iron.options as iron_opts


def main():
    argparser = argparse.ArgumentParser(formatter_class=argparse.ArgumentDefaultsHelpFormatter)
    argparser.add_argument('--eo-file', type=str, help='input export offers file', action='append')
    argparser.add_argument('--eo-dir', type=str, help='dir with export offers files')
    argparser.add_argument('--yt-path', type=str, help='output yt path', required=True)
    argparser.add_argument('--task-type', choices=['perf', 'dyn'], default='perf', help='subj')
    argparser.add_argument('--mode', choices=['local'], default=None,
                           help='"local" for local yt (implies no YQL, 1 cpu, etc)')

    args = argparser.parse_args()

    files_to_upload = []
    if args.eo_file:
        files_to_upload += args.eo_file

    if args.eo_dir:
        for filename in os.listdir(args.eo_dir):
            full_filename = os.path.join(args.eo_dir, filename)
            if os.path.isfile(full_filename):
                files_to_upload.append(full_filename)

    yt_path = args.yt_path

    if args.mode == 'local':
        yt_config = {'proxy': {'url': bannerland.yt_local.get_proxy()}}
        yt_client = yt.YtClient(config=yt_config)
    else:
        yt_client = yt.YtClient(proxy='hahn', config={'token_path': iron_opts.get('yt_token_path')})

    if yt_client.exists(yt_path):
        raise Exception(yt_path + ' exist!')
    yt_client.create('map_node', os.path.dirname(yt_path), ignore_existing=True, recursive=True)
    tmp_yt_dir = '//tmp/upload_export_offers_dir_{0:%Y-%m-%d_%H:%M:%S}_{1}'.format(datetime.datetime.now(), random.random())
    task_yt_dir = yt.ypath_join(tmp_yt_dir, 'tasks')
    offers_yt_dir = yt.ypath_join(tmp_yt_dir, 'offers')
    tao_yt_dir = yt.ypath_join(tmp_yt_dir, 'task_and_offers')

    with yt_client.Transaction():
        uploader = TasksOffersUploader(delete_after=False, yt_client=yt_client)
        uploader.upload(files_to_upload, task_yt_dir, offers_yt_dir)

        joiner = TasksOffersJoiner(mode=args.mode, yt_client=yt_client)
        joiner.merge_to_tao(task_yt_dir, offers_yt_dir, tao_yt_dir)

        finale_yt_tablename = list(filter(lambda x: x.startswith(joiner.get_table_prefix()),
                                          yt_client.list(tao_yt_dir)))[0]
        finale_yt_path = yt.ypath_join(tao_yt_dir, finale_yt_tablename)

        yt_client.move(finale_yt_path, yt_path)

    yt_client.remove(tmp_yt_dir, force=True, recursive=True)
    logging.warning('created table: %s', yt_path)


if __name__ == '__main__':
    main()
