# coding=utf-8
import hashlib
import os
import time
from inspect import getsource
from sys import argv
import textwrap
import timeout_decorator

from yql.api.v1.client import YqlClient
from yql.client.operation import YqlOperationShareIdRequest

import irt.iron.options as iron_opts

import irt.logging


logger = irt.logging.getLogger(None, __name__)

DEFAULT_YT_TOKEN = os.environ.get('YQL_YT_TOKEN_NAME', 'yt_plato')


def get_udf_code(*args):
    """
    input: pairs (function, signature)
    """
    if 'script' in [f.__name__ for f, s in args]:
        raise Exception("Found function called 'script': it is forbidden")

    # generate script
    total_code = '''$script = @@
import json
{}@@;\n'''.format('\n'.join([getsource(f) for f, s in args]))

    # define udfs
    udfs = ["${} = Python::{}('{}', $script)".format(f.__name__, f.__name__, s) for f, s in args]
    total_code += ";\n".join(udfs) + ";\n"
    return total_code


def do_yql(client, query, yt_token_name=DEFAULT_YT_TOKEN, title=None, yt_pool=None, timeout_hour=6, parameters=None,
           transaction_id=None, attached_files=None, syntax_version=1):
    # attached_file - dict{yql_name: local_path} or None - ничего не прикреплять
    # (attached_file только для ya make сборки! без сборки бросает исключение) см. https://a.yandex-team.ru/review/700777/details

    query_prefix = ''
    if client.config.server != 'localhost' and yt_token_name:
        query_prefix += u'PRAGMA yt.Auth = "{}";\n'.format(yt_token_name)

    if yt_pool is not None:
        query_prefix += u'PRAGMA yt.Pool = "{}";\n'.format(yt_pool)

    if not transaction_id and parameters and 'transaction_id' in parameters:
        transaction_id = parameters.pop('transaction_id')
    if transaction_id:
        query_prefix += u'PRAGMA yt.ExternalTx = "{}";\n'.format(transaction_id)

    query = textwrap.dedent(query)
    query = u'{}\n{}'.format(query_prefix, query)

    # append binary/script name to make YQL-caller detectable
    md5_hash = hashlib.md5(query).hexdigest()
    recalc_stamp = ":".join(os.uname() + (str(time.time()),))
    query += "\n-- {}\n-- {}\n".format(argv[0], recalc_stamp)

    prev_title = client.config.current_query_title
    client.config.current_query_title = (title if title else os.path.basename(argv[0])) + ' ' + md5_hash + ' YQL'

    request = client.query(query, syntax_version=syntax_version)
    if attached_files:
        for yql_name, local_path in attached_files.items():
            if not os.path.isfile(local_path):
                logger.error("File "+local_path+"not found!")
                raise OSError("File "+local_path+"not found!")
            request.attach_file(os.path.realpath(local_path), yql_name)

    if parameters is None:
        request.run()
    else:
        request.run(parameters=parameters)

    client.config.current_query_title = prev_title  # restore query title

    shared_id = _get_yql_share_url(request.operation_id)

    logger.error('Yql Operation started: ' + shared_id)

    timeout_error_str = ("ERROR: Yql Operation {} is very long: set good timeout please (now {} hour)"
                         .format(shared_id, timeout_hour))

    timeout_wrapper = timeout_decorator.timeout(timeout_hour * 3600, exception_message=timeout_error_str)
    results = timeout_wrapper(request.get_results)(wait=True)

    if results.is_success:
        return results
    else:
        '''
        # TODO - do not forget about title
        while results.status != 'ERROR':
            sys.stderr.write("Request status: {}\n".format(results.status))
            sys.stderr.write("Errors:\n")
            sys.stderr.write("\n".join(["\t" + str(error) for error in results.errors])+"\n")
            time.sleep(5)
        '''

        raise RuntimeError("Request failed with status: {}\nErrors:\n{}".format(
            results.status,
            "\n".join(["\t" + str(error) for error in results.errors]))
        )


def list_yql_result(result):
    retval = []
    for table in result:
        for row in table.rows:
            retval.append({table.column_names[i]: row[i] for i in range(len(row))})
    return retval


def get_yql_bm_config(db=None):
    # YQL_TOKEN -> YQL_TOKEN_PATH -> BM token
    if 'YQL_TOKEN' in os.environ:
        token = os.getenv('YQL_TOKEN')
    else:
        token_path = os.getenv('YQL_TOKEN_PATH') or iron_opts.get('yql_params')['token_path']
        with open(token_path) as token_f:
            token = token_f.read().rstrip()

    return {
        'db': db or 'hahn',
        'token': token
    }


def _get_yql_share_url(operation_id):
    share_id_request = YqlOperationShareIdRequest(operation_id)
    web_url = 'https://yql.yandex-team.ru'
    share_id_request.run()
    if share_id_request.is_ok:
        return '/'.join([web_url, 'Operations', share_id_request.json])
    else:
        return 'WARNING: YQL Failed to get share link.'


def get_client(db=None):
    yql_client_config = get_yql_bm_config(db)
    return YqlClient(db=yql_client_config['db'], token=yql_client_config['token'])
