import argparse
import logging
import os

import yt.wrapper as yt
from yql.api.v1.client import YqlClient

from lib_monitoring import write_monitoring_to_graphite, get_date, validate_date


logging.basicConfig(level=logging.INFO)


HAHN_DEFAULT = 'hahn'


def get_tables(date):
    stat_table = '//cooked_logs/bs-chevent-cooked-log/1d/' + date
    if not yt.exists(stat_table):
        raise RuntimeError("Stat table {} doesn't exist, exit".format(stat_table))

    campaigns_table = '//home/direct/db-archive/{}/campaigns'.format(date)
    if not yt.exists(campaigns_table):
        raise RuntimeError("Campaigns table {} doesn't exist, exit".format(campaigns_table))
    return stat_table, campaigns_table


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument('--date', help="YYYY-MM-DD. If not specified, yesterday will be used")
    parser.add_argument('--bm-cost-limit-truncate', type=float, default=0.8)
    parser.add_argument('--yt-cluster', default=HAHN_DEFAULT)
    parser.add_argument('--yt-pool', default="broadmatching")
    args = parser.parse_args()

    # noinspection PyUnresolvedReferences
    yt.config["proxy"]["url"] = args.yt_cluster
    yql_token = os.getenv('YQL_TOKEN')

    date = args.date
    stat_table, campaigns_table = None, None
    if date is None:
        for days_back in range(1, 8):
            date = get_date(days_back=days_back)
            try:
                stat_table, campaigns_table = get_tables(date)
            except RuntimeError:
                continue
            else:
                break
        if stat_table is None or campaigns_table is None:
            raise RuntimeError('No appropriate dates found')
    elif not validate_date(date):
        raise ValueError("Incorrect date: {}".format(date))
    else:
        stat_table, campaigns_table = get_tables(date)
    logging.info("Process date {}".format(date))

    yql_client = YqlClient(db=args.yt_cluster, token=yql_token)
    yql_client.config.current_query_title = 'YQL monitoring: cost ' + date

    query = """
    pragma yt.Pool = '{yt_pool}';
    $syn_sd = (
        SELECT
            SimDistance
        FROM `home/yabs/dict/BroadmatchType`
        WHERE BMType = "synonym" and EndDate > DateTime::ToSeconds(DateTime::MakeDatetime(DateTime::ParseIso8601('{date}T00:00:00+03:00')))
    );
    $bm_cost_limit = ($bm_limit, $bm_flag) -> {{
        return if ($bm_flag == 'No', 0., $bm_limit / 100.)
    }};

    $is_online_bm = ($simdistance) -> {{RETURN $simdistance in (25, 26)}};
    $alpha = ($bm_limit, $bm_flag) -> {{RETURN min_of(0.8, $bm_cost_limit($bm_limit, $bm_flag))}};

    from
        `{stat_table}` as STAT
        inner join `{campaigns_table}` as CAMP
        on cast(STAT.orderid as uint64) == CAMP.OrderID
        inner join `home/yabs/dict/Page` as PAGE
        on cast(STAT.pageid as int64) == PAGE.PageID
        left join $syn_sd as SYN
        on cast(STAT.simdistance as uint64) == SYN.SimDistance
    select
        sum_if(STAT.eventcost/1e6, STAT.contexttype == 3 and SYN.SimDistance is null) ?? 0 as BMCost,
        sum_if(STAT.eventcost/1e6, STAT.contexttype == 3 and SYN.SimDistance is not null) ?? 0 as SynonymCost,
        sum_if(STAT.eventcost/1e6, STAT.contexttype == 3 and SYN.SimDistance is null and $is_online_bm(STAT.simdistance)) ?? 0 as OnlineBMCost,
        sum_if(STAT.eventcost/1e6, STAT.contexttype == 3 and SYN.SimDistance is null and not $is_online_bm(STAT.simdistance)) ?? 0 as OfflineBMCost,
        sum_if(STAT.eventcost/1e6, STAT.contexttype == 7) ?? 0 as DynamicCost,
        sum_if(STAT.eventcost/1e6, STAT.contexttype == 11) ?? 0 as RMCost,
        sum_if(STAT.eventcost/1e6, STAT.contexttype == 1) ?? 0 as OrigCost,
        sum(STAT.eventcost/1e6) as Cost,
        sum_if(
            STAT.eventcost/1e6 * $alpha(CAMP.broad_match_limit, CAMP.broad_match_flag) / (1 - $alpha(CAMP.broad_match_limit, CAMP.broad_match_flag)),
            not (STAT.contexttype == 3 and SYN.SimDistance is null)
        ) as PotentialBMCostUpperBound,
        sum($bm_cost_limit(CAMP.broad_match_limit, CAMP.broad_match_flag) * STAT.eventcost/1e6) as PotentialBMCostLowerBound
    where
        STAT.fraudbits == 0
        and STAT.placeid == 542
        and PAGE.TargetType in (0, 2)
    """.format(yt_pool=args.yt_pool,
               stat_table=stat_table, campaigns_table=campaigns_table, date=date)
    request = yql_client.query(query, syntax_version=1)
    request.run()
    request.wait_progress()
    if not request.is_success:
        raise RuntimeError("Something wrong with request for date {}:\n{}".format(
            date, '\n'.join(map(str, request.errors))))

    result = request.dataframe.to_dict()

    for key in result.keys():
        value = float(result[key][0])
        logging.info('Send to graphite: cost.{} = {} ({})'.format(key, value, date))
        write_monitoring_to_graphite('cost.' + key, value, date)


if __name__ == '__main__':
    main()
