#!/usr/bin/python
# -*- encoding: utf-8 -*-

import sys
import logging
import optparse
import socket
import os, re
import ConfigParser

PATH_CONFIG = '/etc/tcprstat-parser.cfg'

class Logger():
    def __init__(self, log_level, log_file):
        self.log_level = log_level
        self.log_file = log_file

    def logger_func(self):
        self.log_level = self.log_level.upper()
        if os.path.isdir(os.path.dirname(self.log_file)):
            # create logger
            self.logger = logging.getLogger('server_file_log')
            self.logger.setLevel(level=getattr(logging, self.log_level))
            # create file handler and set level to LEVEL
            fh = logging.FileHandler(self.log_file, mode='a')
            fh.setLevel(level=getattr(logging, self.log_level))
            # create formatter
            formatter = logging.Formatter("%(asctime)s:%(levelname)s:%(message)s", "%Y-%m-%d %H:%M:%S")
            # add formatter to fh
            fh.setFormatter(formatter)
            # add ch to logger
            self.logger.addHandler(fh)
        else:
            # create logger
            self.logging.critical('Not found %s for logs' % self.log_file)
            self.logger = logging.getLogger('server_string_log')
            self.logger.setLevel(level=getattr(logging, self.log_level))
            # create console handler and set level to LEVEL
            ch = logging.StreamHandler()
            ch.setLevel(level=getattr(logging, self.log_level))
            # create formatter
            formatter = logging.Formatter('%(asctime)s - %(name)s - %(levelname)s - %(message)s')
            # add formatter to ch
            ch.setFormatter(formatter)
            # add ch to logger
            self.logger.addHandler(ch)

class ParseConfigs():
    ''' Класс для парсинга ini и обычных конфигурационных файлов.
        Входные параметры: 
            1. args - путь до конфигурационных файлов.
               Example: 
                   [ main ]
                   par1 = val1
                   par2 = val2
                   [ test ]
                   par3 = val3
                   par4 = val4
                   [ dev ]
                   par3 = NEWval3 
            2. headers - необязательный параметр, указывающий какие блоки нужно брать. Если 
               значение пустое, то берет 'main'.
               Example: 
                   ParseConfigs( '/etc/test.cnf', headers=['main', 'test'] )
                   ParseConfigs( '/etc/test.cnf', headers='main,test' )
        Выходные параметры:
            1. При пустом headers создается атрибут self.config, ссылающийся на дерево параметров конфига
                   { 'main': { 'par1': val1, 'par2': val2 }, 
                     'test': { 'par3': val3, 'par4': val4 }, 
                     'dev': { 'par3': NEWval3 } 
                   }
            2. При указании headers создается атрибут self.config, ссылающийся на словарь агрегированных
               параметров из блоков конфига. Играет порадок расположения headers в self.headers. То что
               старее - перезаписывается новыми параметрами.
               Example:
                   headers=['test']
                   { 'par3': val3, 'par4': val4 }
                   headers=['main', 'test']
                   { 'par1': val1, 'par2': val2, 'par3': val3, 'par4': val4 }
                   headers=['main', 'test', 'dev']
                   { 'par1': val1, 'par2': val2, 'par3': NEWval3, 'par4': val4 }
    '''
    def __init__(self, args, **kwargs):
        self.path_config = args 
        self.headers = kwargs.get('headers', list())
        self.parse_configs()

    def parse_configs(self):
        ''' Функция для парсинга ini и обычных конфигурационных файлов.
            Входящие значения атрибутов self.headers и self.path_config.
            На выходе создает аттрибут self.config.  
        '''
        self.config = dict()
        if os.path.isfile(self.path_config):
            parser = ConfigParser.SafeConfigParser()
            parser.read(self.path_config)
            if not self.__dict__.has_key('headers') or not self.headers: 
                self.config.update(parser._sections)
            else:
                self.headers = self.headers.split(',') if isinstance(self.headers, str) else self.headers
                [ self.config.update(parser._sections.get(_header, dict())) for _header in self.headers ]
    
class TcpRstat(Logger,ParseConfigs):
    def __init__(self, kwargs):
        ''' Удаляем пустые ключи. '''
        [ kwargs.pop(parm) for parm in kwargs.keys()
                           if not kwargs.get(parm) ]
        ''' Получаем список входящих параметров из входящих параметров консоли. '''
        self.__dict__.update(kwargs)
        ''' Читаем конфигурационный файл программы по пути self.path_config. '''
        self.parse_configs()
        ''' Обновляем список конфигураций файла, списком полученным 
            из входящих параметров из консоли.
        '''
        self.config.update(self.__dict__)
        self.log_level = self.config.get('log_level')
        self.log_file = self.config.get('log_file')
        self.logger_func()
        self.logger.debug('Read configs: %s' % str(self.path_config))
        self.logger.debug('Read headers: %s' % str(self.headers))
        self.logger.debug('Update config variables: %s' % str(self.config))
    
    def __call__(self):
        ''' Забираем со стандартного ввода данные приходящие из tcprstat.
            Если получаем пустую строку или какое-то непонятное значение, 
            то выходим из программы.
        '''
        hostname = socket.getfqdn().replace('.', '_')
        names = ['response_max', 'response_max95', 'response_max99', 'request_count']
        names = ['.'.join(['one_min', hostname, self.config.get('graph_name'), _key]) for _key in names ]
        while sys.stdin is not None:
          line = sys.stdin.readline().rstrip('\n')
          try:
              self.timestamp, count, max, max95, max99 = line.rsplit('\t')
              self.timestamp = str(self.timestamp)
              self.logger.debug('Parse string: Unix time:  %s Request:  %s Response max:  %s Response max96:  %s Response max99: %s' % (self.timestamp, count, max, max95, max99))
              values =[ int(max)/1000, int(max95)/1000, int(max99)/1000, count ]
              values = [ str(_values) for _values in values ]
              self.result = dict(zip(names, values))
              self.results_to_sender()
          except Exception, error: 
              self.logger.critical('%s. Exit.' %error)
              sys.exit()

    def parse_data(self):
         ''' Функция-генератор для подготовкм запроса. '''
         for _key in self.result:
             value = ' '.join([_key, self.result.get(_key), self.timestamp])
             yield value

    def results_to_sender(self):
        status = False
        sock = None
        try:
            sock = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
            sock.settimeout(int(self.config['timeout']))
            sock.connect((self.config['address'], int(self.config['port'])))
            self.logger.debug('Connect to %s:%s (timeout: %s)' % (self.config['address'], 
                                                                  self.config['port'], 
                                                                  self.config['timeout']))
            send_count = 0
            for metric in self.parse_data():
                self.logger.debug('Metric send: %s' % metric)
                sock.send(''.join([metric, '\n']))
                send_count += 1
            status = True
            self.logger.debug('Sucess send %d params to sender' % send_count)
        except Exception, err:
            self.logger.warning('Error send data to sender: %s' % err.message)
        finally:
            if sock:
                sock.close()
        self.logger.debug('Send status: %s' % status)

if __name__ == "__main__":
     usage = "usage: parse-tcprstat.py"
     parser = optparse.OptionParser(usage=usage)
     parser.add_option("--log-level",
         action="store", dest="log_level", default='WARNING',
         help="Logging level (Example: DEBUG or CRITICAL)")
     parser.add_option("--address",
         action="store", dest="address", default=None,
         help="Default graphite server address (Example: localhost)")
     parser.add_option("--port",
         action="store", dest="port", default=None,
         help="Default graphite server port (Example: 42000)")
     parser.add_option("--graph-name",
         action="store", dest="graph_name", default=None,
         help="Graph name (Example: catalogia:80 or bmprod:3410)")
     parser.add_option("--path-config",
         action="store", dest="path_config", default=PATH_CONFIG,
         help="Path config file (Example: /etc/tcprstat-parser.cfg) ")
     parser.add_option("--headers",
         action="store", dest="headers", default='main',
         help="Get current headers in config (Example: main,test,dev")
     (options, args) = parser.parse_args()
     if args:
         parser.error("Extra arguments defined.")
     
     a = TcpRstat(options.__dict__)    
     
     while sys.stdin is not None:
         a()
         a.send_status
