import logging
import filelock
import datetime

import yt.wrapper as yt

logging.basicConfig(format='%(asctime)s\t[%(process)d]\t%(message)s', level=logging.INFO)
logger = logging.getLogger(__name__)


def reducer(_, rows):
    yield {column_name: row_value for row in rows for column_name, row_value in row.items()}


def extend_schema(base, extension):
    present_fields = {item['name'] for item in base}

    for item in extension:
        if item['name'] not in present_fields:
            base.append(item)

    return base


def prepare_banners_extended(token, cluster, pool=None):
    logger.info('prepare_banners_extended started')

    banners_table = '//home/direct/export/bm/bm_banners'
    banners_categories_table = '//home/catalogia/catalogia-banners-recategorized-fast'
    banners_extended_table = '//home/catalogia/banners_extended'

    yt_client = yt.YtClient(proxy=cluster, token=token)
    banners_schema = yt.yson.loads(yt_client.get(banners_table + '/@schema', format='yson'))
    banners_categories_schema = yt.yson.loads(yt_client.get(banners_categories_table + '/@schema', format='yson'))

    banners_extended_schema = extend_schema(banners_schema, banners_categories_schema)
    banners_extended_schema = [item for item in banners_extended_schema if item['name'] != 'Categories']

    banners_extended_schema.append({'name': 'mctgs', 'type': 'string'})

    with yt_client.Transaction():
        yt_client.run_reduce(
            reducer,
            [
                banners_table,
                yt.TablePath(banners_categories_table, rename_columns={'Categories': 'mctgs'}),
            ],
            yt.TablePath(banners_extended_table, attributes={'schema': banners_extended_schema}),
            reduce_by='bid',
            spec={
                'pool': pool,
                'reducer': {
                    'enable_input_table_index': False,
                },
            },
        )
        yt_client.set_attribute(banners_extended_table, "upload_time", datetime.datetime.utcnow().isoformat())

        logger.info('prepare_banners_extended done!')


def main():
    with open('/opt/broadmatching/secrets/tokens/yt_plato') as token_file:
        token = token_file.read().strip()
        prepare_banners_extended(token, 'hahn', 'catalogia')


if __name__ == '__main__':
    lock = filelock.FileLock('prepare_banners_extended.pid')
    try:
        with lock.acquire(timeout=0):
            main()
    except filelock.Timeout:
        logger.warning('Another instance of this application currently holds the lock.')
