#!/usr/bin/perl -w

use strict;

use open ':utf8';
use utf8;

use Data::Dumper;
use JSON qw(to_json from_json);

use FindBin;
use lib "$FindBin::Bin/../lib";

use Utils::Sys qw(
    md5int
    mem_usage
    get_file_lock
    release_file_lock
    print_err
    handle_errors
    get_children_pids
);

use BM::BMClient::FCGI_HTTP_Server qw(
    get_fcgi_processes_count
    get_fcgi_pid2state
    get_fcgi_process_elapsed_time
    set_process_state
);
use BM::SolomonClient;

use Utils::Hosts qw(get_curr_host);
use Utils::Common;
use Project;

handle_errors();

get_file_lock()  or do {
    print_err("ERROR: could not get lock!"); # ERROR, т.к. предыдущий запуск должен был успеть доработать за минуту
    exit(0);
};

my $solomon_client = BM::SolomonClient->new();
my $curr_host = get_curr_host();
main();

print_err("Done");

exit(0);

sub main {
    my $proj = Project->new({});
    $proj->log("fcgi-http-server_monitor started");

    my %graphite_out;

    for my $state (qw[ free busy alive ]) {
        my $key = "fcgi.states.$state";
        $graphite_out{$key} = get_fcgi_processes_count($proj, $state);
    }

    my $pid2state = get_fcgi_pid2state($proj) // {};
    my $pid2elapsed = {};
    my $pid2cmdact = {};
    for my $pid (keys $pid2state) {
        my $state = $pid2state->{$pid};
        my ($cmd_act) = map { m/^processing:(.+)/; $1 // () } split /,/, $state;  # "$cmd.$act"
        if ($cmd_act) {
            $graphite_out{"fcgi.cmd.$cmd_act.processing_count"} += 1;

            my $elapsed_time = get_fcgi_process_elapsed_time($proj, $pid);
            $pid2elapsed->{$pid} = $elapsed_time;
            $pid2cmdact->{$pid} = $cmd_act;
            $proj->log("cmd_act: $cmd_act elapsed_time: " . ($elapsed_time // 'UNDEF') . " pid: $pid");
        }
    }

    send_metrics($proj, \%graphite_out);

    my $killed_data = kill_hanging_processes($proj, $pid2elapsed, $pid2cmdact);
    send_metrics($proj, $killed_data);
}

sub kill_hanging_processes {
    my ($proj, $pid2elapsed, $pid2cmdact) = @_;

    my $max_allowed_duration = 60*60; # 60 минут. Таймаут Директа в yml2* - 30 минут. Держать http-запрос дольше 60 минут не хотим (даже если клиент готов ждать вечность)

    my $killed_cmdact2count = {};

    my @old_pids = (grep { $pid2elapsed->{$_} > $max_allowed_duration } sort keys($pid2elapsed));
    # За один запуск убиваем не больше одного процесса.
    if (my $pid = (grep { $pid2cmdact->{$_} } @old_pids)[0]) {

        # TODO Хорошо было бы проверять, что процесс за это время не начал обрабатывать другой запрос
        for my $sig (
            15, # SIGTERM
            9, # SIGKILL
        ) {
            my $cmd_act = $pid2cmdact->{$pid};
            my $elapsed_time = $pid2elapsed->{$pid};

            # Убиваем дочерние процессы
            $proj->do_sys_cmd("pkill -$sig -P $pid || true"); # Если нет дочерних процессов, pkill возвращает exit-code 1

            my $res = kill $sig, $pid;
            $proj->log("Sent kill $sig: cmd_act: $cmd_act elapsed_time: $elapsed_time pid: $pid res: $res");
            sleep 30;

            my $cmd = 'ps aux | grep -v grep | grep -P "^\s*bmclient\s*' . $pid . '\s.*\sperl-fcgi$"';
            my $process_still_exists = `$cmd`;  # TODO Здесь было бы правильно проверять exit-code всех команд. Но grep возвращает 1, если ничего не найдено

            my $children_cmd = "ps xao ppid | grep $pid";
            my $children_still_exist = `$children_cmd`;

            unless ($process_still_exists || $children_still_exist) {
                $killed_cmdact2count->{$cmd_act} += 1;
                set_process_state($proj, $pid, ''); # TODO
                last; # Успешно убили процесс, 
            }
            if ($sig == 9) {
                # Сюда попадать не должны. Если попали сюда, значит, `kill -9` не убил процесс - произошло что-то странное (например, скрипт запущен под неправильным пользователем)
                $proj->log("ERROR: SIGKILL did not work! cmd_act: $cmd_act elapsed_time: $elapsed_time pid: $pid res: " . ($res // 'UNDEF'));
            }
        }
    } elsif (@old_pids) {
        $proj->log("ERROR: All old pids are bad: " . to_json([ $pid2elapsed, \@old_pids, $pid2cmdact ]));
    }

    my %graphite_out = map {( "fcgi.cmd.$_.killed_count" => $killed_cmdact2count->{$_} )} (keys $killed_cmdact2count);
    return \%graphite_out;
}

sub send_metrics {
    my ($proj, $data) = @_;

    $proj->log("Sending new fcgi metrics...");
    for my $key (sort keys %$data) {
        my $value = $data->{$key} // next;

        my @metric_data = split /\./, $key;
        my $sensor_template = {
            cluster => "host_info",
            service => "fcgi_statistics",
            sensor  => $metric_data[1],
            value   => $value,
        };
        if (scalar(@metric_data)  == 3) {
            $solomon_client->push_single_sensor({
                %$sensor_template,
                labels => {
                    host => $curr_host,
                    state => $metric_data[2],
                },
            });
        } elsif (scalar(@metric_data) == 5) {
            $solomon_client->push_single_sensor({
                %$sensor_template,
                labels => {
                    act => $metric_data[3],
                    cmd_name => $metric_data[2],
                    host => $curr_host,
                    property => $metric_data[4],
                },
            });
        }
    }
    $proj->log("send to solomon done");
    return 1;
}

