#!/usr/bin/perl

use strict;
use warnings;
use autodie;
use utf8;

use open ":utf8";
no warnings "utf8";

use FindBin;
use lib "$FindBin::Bin/../lib";
use lib "$FindBin::Bin/../wlib";

use Utils::Sys qw(get_file_lock);
use CatalogiaMediaProject;


get_file_lock() and main();
exit(0);


sub main {
    my $proj = shift // _get_proj();

    $proj->log("Start cleanup...");
    
    # PhraseListCache
    _clean_table(
        $proj,
        dbt => $proj->phrase_list_cache,
        time_field => "greatest(AccessTime, UpdateTime)",
        interval => "3 month",
    );

    # AutoModerationScoreGroupExamples
    _clean_table(
        $proj,
        dbt => $proj->dbtable('AutoModerationScoreGroupExamples'),
        time_field => "UpdateTime",
        interval => "3 month",
    );

    clean_log_tables($proj);
    clean_dblist_tables($proj);
    
    $proj->log("Cleanup done.");
    
    return 1;
}

sub _get_proj {
    return CatalogiaMediaProject->new({
        indcmd => 1,
        no_auth => 1,
        no_form => 1,
        nrmsrv => 0,
    });
}

sub _clean_table {
    my $proj = shift;
    my %params = @_;

    my $dbt = $params{dbt} // die "provide dbt";
    my $interval = $params{interval} // die "provide interval";
    my $time_field = $params{time_field} // die "provide time_field";

    my $table_name = $dbt->db_table;
    
    $proj->log("Cleaning records older than $interval in '$table_name'...");
    
    my $rows = $dbt->Do_SQL("
        delete from $table_name
        where $time_field < now() - interval $interval 
    ");

    $proj->log("Cleaning '$table_name' done, $rows rows deleted");
}

sub clean_log_tables {
    my $proj = shift;

    my $interval = '1 year';
    $proj->log("Cleaning log tables...");

    my $dbh = $proj->catalogia_media_dbh;
    my @tables = map { values %$_ } @{ $dbh->List_SQL("
        show tables like 'Log_%'
    ") };

    $proj->log(scalar(@tables) . " log tables to clean...");
    
    my @check_columns = qw/ID cmd login act acttime/;
    for my $table (@tables) {
        my $present_columns = $dbh->List_SQL("
            show columns from $table
            where Field in (" . join(", ", map { $dbh->dbh->quote($_) } @check_columns) . ")
        ");
        
        if (scalar(@$present_columns) != scalar(@check_columns)) {
            $proj->log("WARN: Table '$table' is not a log table");
            next;
        } else {
            _clean_table(
                $proj,
                dbt => $proj->dbtable($table),
                time_field => 'acttime',
                interval => $interval,
            );

            my $row_count = $dbh->List_SQL("
                select count(*) as Count
                from $table
            ")->[0]->{Count};

            if ($row_count == 0) {
                $proj->log("Table '$table' is empty, drop it");
                $dbh->Do_SQL("
                    drop table $table
                ");
            } else {
                # эти таблицы - MyISAM, им желателен optimize после очистки
                $proj->log("Table '$table' is not empty, optimize it...");
                $dbh->Do_SQL("
                    optimize table $table
                ");
                $proj->log("Table '$table' - optimize done");
            }
        }
    }

    $proj->log("Cleaning log tables done");
}

sub clean_dblist_tables {
    my $proj = shift;

    my $interval = '1 week';
    $proj->log("Cleaning dblist tables older than $interval...");

    my $dbh = $proj->catalogia_media_dbh;
    my @tables = map { $_->{Name} } @{ $dbh->List_SQL("
        show table status
        where
            Name like 'dblist_%'
            and greatest(Create_time, Update_time) < now() - interval $interval;
    ") };

    $proj->log(scalar(@tables) . " old dblist tables, drop them...");

    for my $table (@tables) {
        $proj->log("Drop table '$table'");
        $dbh->Do_SQL("
            drop table $table
        ");
    }

    $proj->log("Cleaning log tables done");
}
