#!/usr/bin/perl -w

use strict;

use utf8;
use open ':utf8';
no warnings 'utf8';
binmode(STDIN,  ":utf8");
binmode(STDOUT, ":utf8");
binmode(STDERR, ":utf8");

use FindBin;
use lib "$FindBin::Bin/../lib";
use Project;
use Getopt::Long qw/GetOptions/;
use List::Util qw/max/;
use Utils::Server;
use Utils::Common;
use Utils::Sys qw/
    get_file_lock
    release_file_lock
    print_log
    print_err
    handle_errors
    save_json
/;

my $server_name = undef;
my $no_restart_time = undef;
my $force_restart_time = undef;
my $log_file = undef;
my $err_file = undef;

GetOptions(
    'server-name=s' => \$server_name,
    'no-restart-time=s' => \$no_restart_time,
    'force-restart-time=s' => \$force_restart_time,
    'log-file=s' => \$log_file,
    'err-file=s' => \$err_file,
) or die($!);

die("ERROR: missed command line argument --server-name\n") if !defined($server_name);
die("ERROR: missed command line argument --no-restart-time\n") if !defined($no_restart_time);
die("ERROR: missed command line argument --force-restart-time\n") if !defined($force_restart_time);
die("ERROR: missed command line argument --log-file\n") if !defined($log_file);
die("ERROR: missed command line argument --err-file\n") if !defined($err_file);

my $start_server_pid_file = $Utils::Common::options->{'dirs'}{'lock'} . "cdict-restart-server-$server_name";

my $proj = Project->new({});
my $client = $proj->{$server_name};
die("ERROR: unknown server name '$server_name'\n") if !$client;

if (!-e $start_server_pid_file) {
    $proj->log("cdict server '$server_name' is not running, do start");
    start_server($start_server_pid_file, $log_file, $err_file);
    run_cdict_server($client, $server_name);
} else {
    my $start_server_time = (stat($start_server_pid_file))[9];
    if (time() - $start_server_time < $no_restart_time) {
        $proj->log("cdict server '$server_name' is fresh enoughf, do nothing");
    } elsif (time() - $start_server_time > $force_restart_time) {
        $proj->log("cdict server '$server_name' didn't restart for long time, do restart");
        stop_cdict_server($client, $server_name);
        stop_server($start_server_pid_file);
        shutdown_server($start_server_pid_file);
        start_server($start_server_pid_file, $log_file, $err_file);
        run_cdict_server($client, $server_name);
    } elsif (get_data_file_modify_time($client) > $start_server_time) {
        $proj->log("cdict server '$server_name' data is outdated, do restart");
        stop_cdict_server($client, $server_name);
        stop_server($start_server_pid_file);
        shutdown_server($start_server_pid_file);
        start_server($start_server_pid_file, $log_file, $err_file);
        run_cdict_server($client, $server_name);
    } else {
        $proj->log("cdict server '$server_name' data is up-to-date, do nothing");
    }
}

sub run_cdict_server {
    my $client = shift;
    my $server_name = shift;

    $proj->log("start cdict server '$server_name'");
    # команда для запуска
    my @cmd = (
        "LD_PRELOAD=\"/usr/lib/libtcmalloc_minimal.so.4\"",
        $client->server_dir."/cdict", 
        "--norm-config=".$client->norm_config_file,
        "--port=".$client->port,
    );

    # выбор входных данных
    if($client->single_file && -e $client->single_file) {
        push @cmd, "--single-file=" . $client->single_file;
    } elsif($client->index_file && -e $client->index_file) {
        push @cmd, "--data-file=".$client->index_file;
        push @cmd, "--dict-file=".$client->dict_file;
    } else {
        die("ERROR: no data file found\n");
    }

    if ($client->info_file) {
        my $mtime = get_data_file_modify_time($client);
        save_json({ mtime => $mtime }, $client->info_file);
    }

    Utils::Sys::do_sys_cmd(join(' ', @cmd));
    $proj->log("/ start cdict server '$server_name'");
}

sub stop_cdict_server {
    my $client = shift;
    my $server_name = shift;

    $client->{'host'} = '127.0.0.1';
    $proj->log("stop cdict server '$server_name'");
    $client->stop_server;
    $proj->log("/ stop cdict server '$server_name'");
}

sub get_data_file_modify_time {
    my $client = shift;

    my @data_files = ();
    if($client->single_file && -e $client->single_file) {
        push @data_files, $client->single_file;
    } elsif($client->index_file && -e $client->index_file) {
        push @data_files, $client->index_file;
        push @data_files, $client->dict_file;
    } else {
        die("ERROR: no data file found\n");
    }
    return max(map {(stat($_))[9]} @data_files);
}
