#include "BenderServer.h"

#include "Index.h"

#include <rt-research/broadmatching/scripts/cpp-source/common/Session.h>

#include <library/cpp/logger/global/global.h>

#include <util/datetime/cputimer.h>
#include <util/generic/hash.h>
#include <util/string/builder.h>
#include <util/string/join.h>
#include <util/string/split.h>

BenderServer::BenderServer(const Index& ind, int port) : Server(port), index(ind) {
}

TString prepare_output(const std::vector<DataItem>& v) {
    TVector<TString> output;
    for(const auto& el: v) {
        output.push_back("");
        output.push_back(ToString(el));
    }

    return JoinSeq("\t", output);
}

void BenderServer::ExecCommand(const TStringBuf& command, Session* session) {
    static const THashMap<TString, ui8> arg_counts = {
        { "bye", 0 },
        { "stop", 0 },
        { "top", 3 },
        { "random", 1 },
        { "random_query", 2 },
    };

    TVector<TStringBuf> cmds;
    Split(command, "/", cmds);

    if(!cmds) {
        ERROR_LOG << "Empty command list";
        session->SendResponse("ERROR: empty command");
        return;
    }

    std::vector<std::string> result;
    for(const auto& command: cmds) {
        const TVector<TString> cmd = StringSplitter(command).Split('\t');

        if(!cmd.size()) {
            ERROR_LOG << "Empty command";
            session->SendResponse("ERROR: empty command");
            return;
        }

        INFO_LOG << session->getClientId() << ":" << JoinSeq(":", cmd);

        TSimpleTimer timer;
        if(auto it = arg_counts.FindPtr(cmd[0])) {
            if(cmd.size() != (*it + 1)) {
                ERROR_LOG << "Wrong arguments count. Command `" << cmd[0] << "` takes " << *it << " arguments, but got " << cmd.size() << " arguments";
                session->SendResponse((TStringBuilder{} << "ERROR: '" << cmd[0] << "' takes " << *it << " arguments").Data());
                return;
            }
        } else {
            ERROR_LOG << "Unknown command: " << cmd[0];
            session->SendResponse((TStringBuilder{} << "ERROR: unknown command '" << cmd[0] << "'").Data());
            return;
        }

        if(cmd[0] == "bye") {
            INFO_LOG << "Stop current session";
            session->SendResponse("BYE");
            session->Stop();
        } else if(cmd[0] == "stop") {
            INFO_LOG << "Stop server";
            session->SendResponse("BYE");
            session->Stop();
            Stop();
        } else if(cmd[0] == "top") {
            INFO_LOG << "Get top " << cmd[1] << " queries with pattern `" << cmd[3] << "`";

            std::vector<DataItem> bids;
            index.Search(cmd[3].Data(), FromString<unsigned>(cmd[1]), bids);
            result.push_back(prepare_output(bids));

        } else if (cmd[0] == "random") {
            INFO_LOG << "Get random banners: " << cmd[1];

            std::vector<DataItem> bids;
            index.RandomBanners(FromString<unsigned>(cmd[1]), bids);
            result.push_back(prepare_output(bids));

        } else if (cmd[0] == "random_query") {
            INFO_LOG << "Get random " << cmd[1] << " banners with word `" << cmd[2] << "`";

            std::vector<DataItem> bids;
            index.RandomBannersQuery(cmd[2].Data(), FromString<unsigned>(cmd[1]), bids);
            result.push_back(prepare_output(bids));

        }

        INFO_LOG << session->getClientId() << ":" << JoinSeq(":", cmd) << ":" << timer.Get().MilliSeconds() << "ms";
    }

    session->SendResponse(JoinSeq("/", result).c_str());
}
