#include "Index.h"

#include "Pattern.h"
#include "Query.h"

#include <library/cpp/logger/global/global.h>

#include <util/generic/maybe.h>
#include <util/system/mem_info.h>

#include <fstream>
#include <unordered_map>

const double SEARCH_TIMEOUT = 5.0;

const char * const BENDER_VERSION = "BENDER01";

struct BinaryFileHeader {
    char    version[9];        // BENDER_VERSION
    size_t  bids_size;
    size_t  words_data_size;
    size_t  words_index_size;
};

bool CompareWords2(const char *lhs, const char *rhs) {
    return strcmp(lhs, rhs) < 0;
}

bool Index::LoadText(const char *encoded_file, const char *index_file) {
    bool encoded_ok = true;

    encoded_ok = LoadEncoded(encoded_file, true) && LoadEncoded(encoded_file, false);

    return encoded_ok &&
        LoadIndex(index_file, true) &&
        LoadIndex(index_file, false);
}

bool Index::LoadEncoded(const char *encoded_file, bool is_reserving) {
    std::ifstream stream;
    std::string buffer;
    size_t bids_count = 1, i = 0;

    INFO_LOG << "Index::LoadEncoded(" << is_reserving << ") mem usage: " << NMemInfo::GetMemInfo().VMS;

    if(!is_reserving) {
        // пустой текст с номером 0
        bids[i++] = 0;
    }

    stream.open(encoded_file);
    if(stream.fail()) {
        return false;
    }
    while(stream.good()) {
        std::getline(stream, buffer);

        if (buffer.length() == 0) {
            continue;
        }

        if(is_reserving) {
            // подсчёт объёма
            bids_count++;
        } else {
            // заполнение структур данных
            bids[i++] = atol(buffer.c_str());
        }
    }
    stream.close();

    if(is_reserving) {
        // резервирование памяти
        INFO_LOG << "mem usage before reserving: " << NMemInfo::GetMemInfo().VMS;
        INFO_LOG << "bids_count: " << bids_count;
        bids.resize(bids_count);
    }

    INFO_LOG << "/ Index::LoadEncoded(" << is_reserving << ") mem usage: " << NMemInfo::GetMemInfo().VMS;

    return true;
}

bool Index::LoadIndex(const char *index_file, bool is_reserving) {
    std::ifstream stream;
    std::string buffer;
    size_t words_size = 1 + sizeof(bid_size), words_count = 1;
    char *p;

    INFO_LOG << "Index::LoadIndex(" << is_reserving << ") mem usage: " << NMemInfo::GetMemInfo().VMS;

    if(!is_reserving) {
        p = (char*)(((bid_size*)(&words_data[1])) + 1);

        // пустое слово с номером 0
        words_data[0] = 0;
        *((bid_size*)(&words_data[1])) = 0;
        words_index.push_back(&words_data[0]);
    }

    stream.open(index_file);
    if(stream.fail()) {
        return false;
    }
    std::string last_word = "";
    while(stream.good()) {
        std::getline(stream, buffer);

        char *end = &buffer[0];
        while(*end && *end != '\t') {
            end++;
        }
        if(!*end) {
            continue;
        }

        std::string word(&buffer[0], end);
        end++;
        if(!*end) {
            continue;
        }

        if (word.compare(last_word) != 0) {
        // записываем текущее слово
            if(is_reserving) {
                words_size += word.size() + 1;
                words_count++;
            } else {
                words_index.push_back(p);
                strcpy(p, word.c_str());
                p += word.size() + 1;
            }
        } else {
            if (is_reserving) {
                words_size -= sizeof(bid_size);
            } else {
                // возвращаемся обратно, чтоб затереть 0
                p -= sizeof(bid_size);
            }
        }

        // парсим последовательность айдишников
        bid_size curr_id = 0;
        bool is_done = false;
        while(!is_done) {
            if(*end >= '0' && *end <= '9') {
                curr_id = curr_id * 10 + (*end - '0');
            } else {
                if(is_reserving) {
                    words_size += sizeof(bid_size);
                } else {
                    *((bid_size*)p) = curr_id;
                    p += sizeof(bid_size);
                }

                curr_id = 0;
            }

            if(!*end) {
                is_done = true;
            } else {
                end++;
            }
        }

        // последовательность айдишников заканчивается нулём
        if(is_reserving) {
            words_size += sizeof(bid_size);
        } else {
            *((bid_size*)p) = 0;
            p += sizeof(bid_size);
        }

        last_word = word;
    }
    stream.close();

    if(is_reserving) {
        // резервирование памяти
        INFO_LOG << "mem usage before reserving: " << NMemInfo::GetMemInfo().VMS;
        INFO_LOG << "words_size: " << words_size;
        INFO_LOG << "words_count: " << words_count;
        words_data.resize(words_size);
        words_index.reserve(words_count);
    }

    INFO_LOG << "/ Index::LoadIndex(" << is_reserving << ") mem usage: " << NMemInfo::GetMemInfo().VMS;

    return true;
}

bool Index::SaveBinary(const char *file_name) const {
    std::ofstream stream(file_name, std::ios::binary);
    size_t i;

    if(stream.fail()) {
        return false;
    }

    // заголовок
    BinaryFileHeader header;
    strcpy(header.version, BENDER_VERSION);
    header.bids_size        = bids.size();
    header.words_data_size  = words_data.size();
    header.words_index_size = words_index.size();
    stream.write((const char*)&header, sizeof(header));

    // словарь текстов
    stream.write((const char*)(&bids[0]), bids.size() * sizeof(DataItem));

    // индекс словопозиций
    stream.write(&words_data[0], words_data.size());
    for(i = 0; i < words_index.size(); i++) {
        size_t offset = words_index[i] - &words_data[0];
        stream.write((const char*)(&offset), sizeof(size_t));
    }

    return true;
}

bool Index::LoadBinary(const char *file_name) {
    std::ifstream stream(file_name, std::ios::binary);
    size_t i;

    if(stream.fail()) {
        return false;
    }

    // заголовок
    BinaryFileHeader header;
    stream.read((char *) &header, sizeof(header));

    if(strcmp(header.version, BENDER_VERSION)) {
        return false;
    }

    bids.resize(header.bids_size);
    words_data.resize(header.words_data_size);
    words_index.resize(header.words_index_size);

    // словарь текстов
    stream.read((char*)(&bids[0]), bids.size() * sizeof(DataItem));

    // индекс словопозиций
    stream.read(&words_data[0], words_data.size());
    for(i = 0; i < words_index.size(); i++) {
        size_t offset;
        stream.read((char*)(&offset), sizeof(size_t));
        words_index[i] = &words_data[offset];
    }

    return true;
}

void Index::Dump() const {
    INFO_LOG << bids.size() << " bids (" << bids.size() * sizeof(DataItem) << " bytes)";
    INFO_LOG << words_index.size() << " words (" << words_data.size() << " bytes)";
}

size_t Index::FindWord(const char *word) const {
    std::vector<char*>::const_iterator it = std::lower_bound(words_index.begin(), words_index.end(), word, CompareWords2);

    if(it != words_index.end() && !strcmp(word, *it)) {
        return it - words_index.begin();
    }

    return 0;
}

// получить указатель на то место в индексе, где для слова с номером word_i начинается список баннеров
const bid_size* Index::GetWordIndex(bid_size word_i) const {
    const char *p;

    for(p = words_index[word_i]; *p; p++) ;

    return (const bid_size*)(p + 1);
}

// получить указатель на то место в индексе, где для слова с номером word_i кончается список баннеров
const bid_size* Index::GetWordIndexEnd(bid_size word_i) const {

    const bid_size* end = nullptr;

    // мы не храним число баннеров у слова явно, и теоретически должны бы проходиться по списку линейно, как strlen
    // но у слов вроде lang_ru этот список очень длинный, и ходить по нему на каждый запрос - не дело
    // поэтому мы пользуемся тем, что и в words_data, и в words_index слова отсортированы
    // т.е. указатель на начало следующего слова - это одновременно указатель на позицию сразу после того места, где кончаются данные для текущего слова

    if (word_i == (words_index.size() - 1)) {
        // особая логика для последнего слова в индексе: для него конец определяем не по началу следующего слова, а по концу всего массива данных
        end = (const bid_size*) (words_data.data() + words_data.size());
    } else {
        end = (const bid_size*) words_index[word_i + 1];
    }

    // список баннеров заканчивается нулём, т.е. нужно вычесть ещё единицу
    end--;

    return end;
}

void Index::Search(const char* phrase, unsigned max_top, std::vector<DataItem>& result) const {
    Pattern pattern(phrase);
    bid_size infty = (bid_size)-1;

    std::vector<const bid_size*> pos(pattern.NumWords());
    for(unsigned i = 0; i < pattern.NumWords(); i++) {
        bid_size word_i = FindWord(pattern.Word(i));  // получить индекс слова в words_index
        pos[i] = GetWordIndex(word_i);  // получить указатель на то место в words_data, где для этого слова начинается список номеров баннеров
    }

    // pattern.Match(mask) умеет сказать, достаточно ли заданного набора слов, чтобы соответствовать запросу
    // mask - это вектор [номер слова в запросе] => [есть ли слово в наборе]

    // для начала проверим, что совокупного набора слов, у которых есть какие-то баннеры, достаточно чтобы соответсвовать запросу
    // если нет - дальше можно ничего не делать, любой другой набор что мы проверим будет заведомо у́же

    std::vector<bool> test_set(pattern.NumWords());
    for(unsigned j = 0; j < pattern.NumWords(); j++) {
        test_set[j] = (*pos[j] != 0);
    }
    if(pattern.Match(test_set) == "") {
        return;
    }

    bid_size minq = 0;
    bool is_done = false;

    if(max_top) {
        result.reserve(max_top);
    }

    // pos: вектор, в котором по номеру слова в запросе лежит указатель на отсортированный список номеров баннеров, в которых есть это слово, заканчиващийся нулём
    // minq: номер баннера в рассмотрении

    // на каждой итерации мы:
    // 1) проверяем, не подходит ли баннер с номером minq под запрос
    // 2) сдвигаем minq настолько вправо, насколько можем
    // 3) для тех указателей в pos, у которых баннер с наименьшим номером равен minq, сдвигаем и их насколько можем

    timespec start, finish;
    double delta;
    clock_gettime(CLOCK_MONOTONIC, &start);
    while(!is_done && (!max_top || (max_top > result.size()))) {
        // если считаем слишком долго -- выходим
        clock_gettime(CLOCK_MONOTONIC, &finish);
        delta = (finish.tv_sec - start.tv_sec) + (finish.tv_nsec - start.tv_nsec) / 1000000000.0;
        if(delta > SEARCH_TIMEOUT) {
            break;
        }

        // маска слов для текущего баннера в рассмотнии
        std::vector<bool> mask(pattern.NumWords());
        for(unsigned i = 0; i < pos.size(); i++) {
            mask[i] = (*pos[i] && *pos[i] == minq);
        }

        // проверяем, не подходит ли нам текущий баннер
        std::string matched = pattern.Match(mask);
        bool is_matched = !matched.empty();
        if(is_matched) {
            result.push_back(bids[minq]);
        }

        bid_size second_minq = infty;  // правая граница, до которой можно будет сдвинуть указатели в pos, если баннер не подошёл
        if(pattern.IsSimple()) {
            // запрос без скобок
            second_minq = minq + 1;
            for(unsigned i = 0; i < pos.size(); i++) {
                if(*pos[i]) {
                    second_minq = std::max(second_minq, *pos[i]);
                }
            }
        } else if(!is_matched) {
            // запрос со скобками - ищем границу с учётом групп:
            // для каждой группы ищем минимальный номер баннера, который на эту группу сматчится
            // чтобы сматчился весь запрос, нужно чтобы матчились все обязательные группы, т.е. указатели можно сдвигать до максимального из минимумов обязательных групп - никакой баннер меньше заведомо не подойдёт
            std::vector<bid_size> group_min_query(pattern.NumGroups());
            std::fill(group_min_query.begin(), group_min_query.end(), infty);
            for(unsigned i = 0; i < pos.size(); i++) {
                bid_size q = *pos[i];
                if (q == minq && minq != 0) {
                    q = *(pos[i] + 1);
                }
                if (q) {
                    unsigned g = pattern.WordToGroup(i);
                    if (!pattern.IsGroupOptional(g)) {
                        group_min_query[g] = std::min(q, group_min_query[g]);
                    }
                }
            }
            second_minq = minq + 1;
            for(unsigned i = 0; i < pattern.NumGroups(); i++) {
                bid_size q = group_min_query[i];

                if(q < infty) {
                    second_minq = std::max(second_minq, q);
                }
            }
        }

        // сдвигаем влево minq и индексы в pos
        bid_size next_minq = infty;
        for(unsigned i = 0; i < pos.size(); i++) {
            if(!*pos[i]) {
                continue;
            }

            bid_size currq = *pos[i];
            if(currq == minq) {
                pos[i]++;
            }

            if(!is_matched) {
                while(*pos[i] && *pos[i] < second_minq) {
                    pos[i]++;
                }
            }

            if(!*pos[i]) {
                // дошли до конца по одному из индексов -- проверяем, есть ли смысл идти дальше
                std::vector<bool> test_set(pattern.NumWords());
                for(unsigned j = 0; j < pattern.NumWords(); j++) {
                    test_set[j] = *pos[j];
                }
                is_done = (pattern.Match(test_set) == "");
            } else {
                next_minq = std::min(next_minq, *pos[i]);
            }
        }

        minq = next_minq;
    }
}


// https://femida.yandex-team.ru/problems/230 :-)

template<typename T>
class RandomSampler {
public:

    RandomSampler(const T* _data_ptr, size_t _data_size)
        : data_ptr(_data_ptr)
        , data_size(_data_size)
        , cur_data_size(_data_size)
    { }

    TMaybe<T> Choose() {
        if (cur_data_size == 0) {
            return TMaybe<T>();
        }

        size_t i = rand() % cur_data_size;
        T value = data_ptr[i];

        const auto it_from = index_mapping.find(i);
        if (it_from != index_mapping.end()) {
            value = data_ptr[it_from->second];
        }

        size_t map_cur_element_to = cur_data_size - 1;
        const auto it_to = index_mapping.find(map_cur_element_to);
        if (it_to != index_mapping.end()) {
            map_cur_element_to = it_to->second;
        }

        index_mapping[i] = map_cur_element_to;
        cur_data_size--;

        return value;
    }

private:
    const T* data_ptr;
    const size_t data_size;

    size_t cur_data_size;
    std::unordered_map<size_t, size_t> index_mapping;
};

void Index::RandomBanners(unsigned sample_size, std::vector<DataItem>& result) const {
    // bids начинается с нулевого баннера, поэтому в RandomSampler передаём указатель со сдвигом на одну позицию
    RandomSampler<DataItem> sampler(bids.data() + 1, bids.size() - 1);
    while (result.size() < sample_size) {
        auto item = sampler.Choose();
        if (item.Defined()) {
            result.push_back(item.GetRef());
        } else {
            break;
        }
    }
}

// узнать число баннеров у слова с индексом word_i в words_data
bid_size Index::GetBannerCount(bid_size word_i) const {

    const bid_size* begin = GetWordIndex(word_i);
    const bid_size* end = GetWordIndexEnd(word_i);

    assert(begin <= end);

    return end - begin;
}

void Index::RandomBannersQuery(const char* phrase, unsigned sample_size, std::vector<DataItem>& result) const {

    // сейчас мы не поддерживаем здесь полного синтаксиса запросов к bender
    Pattern pattern(phrase);
    if (!pattern.IsSimple()) {
        return;
    }

    // для запросов без токенов есть другой метод
    if (pattern.NumWords() == 0) {
        return;
    }

    // сохраняем позиции всех слов в индексе
    std::vector<bid_size> pos(pattern.NumWords());
    for(unsigned i = 0; i < pattern.NumWords(); i++) {
        bid_size cur_pos = FindWord(pattern.Word(i));  // получить индекс слова в words_index
        if (cur_pos == 0) {
            // такого слова в индексе нет, значит баннеров под этот запрос тоже нет
            return;
        }
        pos[i] = cur_pos;
    }

    // для каждого слова мы узнаём, сколько для него есть баннеров, и выбираем минимальное
    unsigned min_word_i = 0;
    bid_size min_word_banner_count = GetBannerCount(pos[min_word_i]);
    for (unsigned i = 1; i < pattern.NumWords(); i++) {
        bid_size cur_banner_count = GetBannerCount(pos[i]);
        if (cur_banner_count < min_word_banner_count) {
            min_word_i = i;
            min_word_banner_count = cur_banner_count;
        }
    }

    // далее выбираем случайные баннеры из списка баннеров для минимального слова, проверяя для каждого, что оно есть и во всех других словах
    auto min_word_banners_ptr = GetWordIndex(pos[min_word_i]);
    RandomSampler<bid_size> sampler(min_word_banners_ptr, min_word_banner_count);
    while (result.size() < sample_size) {
        auto item = sampler.Choose();
        if (!item.Defined()) {
            // баннеры закончились
            break;
        }

        // для каждого из слов, проверяем что этот баннер в нём есть
        bool ok = true;
        for (unsigned i = 0; i < pattern.NumWords(); i++) {
            if (i == min_word_i) {
                // нет смысла проверять для слова, из которого мы делаем сэмплинг
                continue;
            }

            auto begin = GetWordIndex(pos[i]);
            auto end = GetWordIndexEnd(pos[i]);
            auto it = std::lower_bound(begin, end, item.GetRef());
            if (it == end || *it != item.GetRef()) {
                ok = false;
                break;
            }
        }

        if (ok) {
            result.push_back(bids[item.GetRef()]);
        }
    }
}


