#include "CdictServer.h"

#include "MultiIndex.h"

#include <rt-research/broadmatching/scripts/cpp-source/common/norm/NormDict.h>
#include <rt-research/broadmatching/scripts/cpp-source/common/Session.h>

#include <library/cpp/logger/global/global.h>

#include <util/string/split.h>

#include <climits>
#include <cstdio>
#include <cstdlib>
#include <iostream>

CdictServer::CdictServer(const MultiIndex& ind, const NormDict& nrm, int port) : Server(port), index(ind), norm(nrm) {
}

void CdictServer::PreprocessText(TString& text) {
    for(auto& ch: text) {
        if(ch == '-' || ch == '!') {
            ch = ' ';
        }
    }
}

void CdictServer::GetDictValue(const char *dict, const char *key, TStringBuilder& buffer) {
    IndexValueType value_type;
    const unsigned char* value_ptr;
    const Index* index_ptr;
    index.FindValue(dict, key, value_type, value_ptr, index_ptr);

    if(value_ptr) {
        buffer << "YES\t" << index_ptr->GetValue(value_ptr, value_type);
    } else {
        buffer << "NO";
    }
}

void CdictServer::ExecCommand(const TStringBuf& command, Session* session) {
    std::vector<std::string> words;
    std::vector<unsigned> counts;
    TStringBuilder buffer;

    unsigned status_code;
    decltype (std::chrono::high_resolution_clock::now()) start_time;

    if (logging_level == Full) {
        start_time = std::chrono::high_resolution_clock::now();
    }

    TVector<TStringBuf> cmds;
    Split(command, "/", cmds);

    status_code = 0;
    if(!cmds.size()) {
        // пустая команда -- ошибка
        session->SendResponse("ERROR: empty command");
        status_code = 1;
    }

    for(unsigned i = 0; i < cmds.size() && status_code == 0; ++i) {
        TVector<TString> cmd = StringSplitter(cmds[i]).Split('\t');

        if(!cmds.size()) {
            // пустая команда -- ошибка
            session->SendResponse("ERROR: empty command");
            status_code = 1;
            break;
        } else if(cmd[0] == "bye") {
            // закончить данную сессию
            session->SendResponse("BYE");
            session->Stop();
        } else if(cmd[0] == "stop") {
            // остановить сервер
            session->SendResponse("BYE");
            session->Stop();
            Stop();
        } else if(cmd[0] == "get") {
            // get D K
            // значение ключа K из словаря D
            if(cmd.size() != 3) {
                session->SendResponse("ERROR: 'get' takes 2 arguments");
                status_code = 2;
                break;
            }

            GetDictValue(cmd[1].c_str(), cmd[2].c_str(), buffer);
        } else if(cmd[0] == "getnorm") {
            // get D K [L]
            // значение ключа norm_phr(K) из словаря D, L -- язык
            if(cmd.size() != 3 && cmd.size() != 4) {
                session->SendResponse("ERROR: 'getnorm' takes 2 or 3 arguments");
                status_code = 3;
                break;
            }

            PreprocessText(cmd[2]);
            std::string norm_phr = norm.Normalize(cmd[2].c_str(), (cmd.size() > 3 ? cmd[3].c_str() : "ru"), true);

            GetDictValue(cmd[1].c_str(), norm_phr.c_str(), buffer);
        } else if(cmd[0] == "getnormq") {
            // get D K
            // значение ключа norm_phr(K) из словаря D
            // если norm_phr(K)=="", то возвращает прочерк
            if(cmd.size() != 3 && cmd.size() != 4) {
                session->SendResponse("ERROR: 'getnormq' takes 2 or 3 arguments");
                status_code = 4;
                break;
            }

            PreprocessText(cmd[2]);
            std::string norm_phr = norm.Normalize(cmd[2].c_str(), (cmd.size() > 3 ? cmd[3].c_str() : "ru"), true);

            if(norm_phr.size()) {
                GetDictValue(cmd[1].c_str(), norm_phr.c_str(), buffer);
            } else {
                buffer << "YES\t-";
            }
        } else if(cmd[0] == "getsnorm") {
            // get D K [L]
            // значение ключа snorm_phr(K) из словаря D, L -- язык
            if(cmd.size() != 3 && cmd.size() != 4) {
                session->SendResponse("ERROR: 'getsnorm' takes 2 or 3 arguments");
                status_code = 5;
                break;
            }

            PreprocessText(cmd[2]);
            std::string norm_phr = norm.Snormalize(cmd[2].c_str(), (cmd.size() > 3 ? cmd[3].c_str() : "ru"));
            GetDictValue(cmd[1].c_str(), norm_phr.c_str(), buffer);
        } else if(cmd[0] == "bm" || cmd[0] == "bmf") {
            // bm D N F K
            // D -- имя словаря;
            // N -- максимальное кол-во фраз;
            // F -- флаги;
            // K -- фраза.
            if(cmd.size() != 5) {
                session->SendResponse("ERROR: wrong 'bm' format");
                status_code = 6;
                break;
            }

            size_t max_phrases = atoi(cmd[2].c_str());
            int flags = atoi(cmd[3].c_str());

            // удаление лишних символов и нормализация
            for(size_t char_index = 0; char_index < cmd[4].size(); char_index++) {
                char ch = cmd[4][char_index];
                if(ch == '+' || ch == '!') {
                    cmd[4][char_index] = ' ';
                }
            }
            words.clear();
            norm.GetNormWords(cmd[4].c_str(), "ru", words);

            bool is_bad = false;
            if(flags & BM_CHECK_BAD_WORDS) {
                for(size_t i = 0; !is_bad && i < words.size(); i++) {
                    if(norm.IsNormBad(words[i].c_str())) {
                        is_bad = true;
                    }
                }
            }

            if(!is_bad) {
                if(cmd[0] == "bmf") {
                    matcher.MatchFull(index, cmd[1].c_str(), max_phrases, flags, words, buffer);
                } else {
                    std::sort(words.begin(), words.end());

                    // веса слов
                    counts.resize(words.size());
                    for(size_t word_index = 0; word_index < words.size(); word_index++) {
                        // слова из цифр считаем самыми широкими
                        bool is_number = true;
                        for(size_t char_index = 0; is_number && char_index < words[word_index].size(); char_index++) {
                            char ch = words[word_index][char_index];
                            if(!isdigit(ch) && ch != '-') {
                                is_number = false;
                            }
                        }

                        counts[word_index] = is_number ? UINT_MAX : norm.GetNormCount(words[word_index].c_str());
                    }

                    if(flags & BM_FULL) {
                        matcher.MatchFullWeighted(index, cmd[1].c_str(), max_phrases, flags, words, counts, buffer);
                    } else if(counts.size() > 1 || (counts.size() && counts[0] < UINT_MAX)) {
                        // слишком широкий запрос не пытаемся матчить
                        matcher.Match(index, cmd[1].c_str(), max_phrases, flags, words, counts, buffer);
                    }
                }
            }
        } else {
            // неизвестная команда
            session->SendResponse(("ERROR: unknown command '" + cmd[0] + "'").c_str());
            status_code = 7;
            break;
        }

        // если это не последняя команда, добавляем разделитель
        if(i + 1 < cmds.size()) {
            buffer << "/";
        }
    }
    if (status_code == 0) {
        buffer << "\n";
        session->SendPartResponse(buffer);
    }
    if (logging_level == Full) {
        auto end = std::chrono::high_resolution_clock::now();
        std::string duration = std::to_string( std::chrono::duration_cast<std::chrono::microseconds>(end-start_time).count() / (float)1e6 );
        auto comma_pos = duration.find(',');
        if (comma_pos != std::string::npos) {
            duration[comma_pos] = '.';
        }

        INFO_LOG << "ADDR: " << session->getClientAddr() << " LENGTH:" << command.Size() << " NPHRASES:" << cmds.size() << " STATE:" << status_code << " TIME:" << duration;
    }
}
