#include "Index.h"

#include <library/cpp/logger/global/global.h>

#include <util/datetime/cputimer.h>
#include <util/generic/string.h>
#include <util/string/split.h>
#include <util/system/mem_info.h>

typedef unsigned char CompressedSize;
const CompressedSize MAX_COMPRESSED_SIZE = (~((CompressedSize)0));

const size_t NUM_THREADS = 24;
const size_t BUFFER_SIZE = 1024 * 1024;  // 1 Mb

struct IndexFileHeader {
    char        signature[6];   // "INDEX"
    char        version;        // 0
    unsigned    num_dicts;
    size_t      data_size;
};

struct IndexFileDictV0 {
    unsigned        name_length;
    size_t          num_items;
};

struct IndexFileDictV1 {
    unsigned        name_length;
    size_t          num_items;
    DictPropertiesV1  properties;
};

struct IndexFileDict {
    unsigned        name_length;
    size_t          num_items;
    DictProperties  properties;
};

Index::Index(bool debug, const HuffmanTree& ht) : is_debug(debug), huffman(ht) {
}

bool _CompareKeys(const unsigned char* lhs, const unsigned char* rhs) {
    CompressedSize lsize = *((CompressedSize*)lhs);
    CompressedSize rsize = *((CompressedSize*)rhs);

    return std::lexicographical_compare(
        lhs + sizeof(CompressedSize),
        lhs + sizeof(CompressedSize) + lsize,
        rhs + sizeof(CompressedSize),
        rhs + sizeof(CompressedSize) + rsize
    );
}

bool Index::Load(const char* file_name) {
    INFO_LOG << "Index::Load started (mem: " << NMemInfo::GetMemInfo().VMS << ")";

    if(!InitDicts(file_name)) {
        return false;
    }

    std::ifstream stream(file_name);

    if(stream.fail()) {
        return false;
    }

    char buffer[BUFFER_SIZE];
    unsigned char* curr_ptr = &full_data[0];
    unsigned char* end_ptr = &(full_data.back());
    while(stream.good()) {
        stream.getline(buffer, BUFFER_SIZE);

        if(buffer[0]) {
            TVector<TString> data = StringSplitter(TStringBuf(buffer, strlen(buffer))).Split('\t');
            if(data.size() != 3) {
                INFO_LOG << "bad line " << buffer;
                continue;
            }

            // сжатия ключа и значения
            DictsProperties::iterator dp = dicts_properties.find(data[0]);
            unsigned len = CompressKey(
                data[1].c_str(),
                curr_ptr + sizeof(CompressedSize),
                end_ptr,
                dp->second.key_type);
            if(len > MAX_COMPRESSED_SIZE) {
                INFO_LOG << "key '" << data[1] << "' is too big";
            } else {
                dicts[data[0]].push_back(curr_ptr);
                *((CompressedSize*) curr_ptr) = (CompressedSize)len;
                curr_ptr += len + sizeof(CompressedSize);
                curr_ptr += CompressValue(data[2], curr_ptr, end_ptr, dp->second.value_type);
            }
        }
    }

    for(auto it = dicts.begin(); it != dicts.end(); it++) {
        std::sort(it->second.begin(), it->second.end(), _CompareKeys);
    }

    INFO_LOG << "Index::Load finished (mem: " << NMemInfo::GetMemInfo().VMS << ")";

    return true;
}


bool Index::LoadParallel(const char* file_name) {
    INFO_LOG << "Index::LoadParallel started (mem: " << NMemInfo::GetMemInfo().VMS << ")";

    // создаем SizePrecalculator
    SizePrecalculator size_prec(*this, NUM_THREADS, BUFFER_SIZE);
    // устанавливаем заранее полученные свойства словарей
    size_prec.set_dicts_properties(dicts_properties);

    /* инициализируем словари в параллельном режиме */
    TSimpleTimer timer;

    // вычисление размеров итоговых данных и словарей
    // результат заносится в size_prec
    if(!InitDictsParallel(file_name, size_prec))
        return false;

    INFO_LOG << "dicts initialized in " << timer.Get().Seconds() << " sec"
           << " mem(" << NMemInfo::GetMemInfo().VMS << ")";

    /* сжатие данных */
    timer.Reset();

    // подсчитывам смещения для записи результатов каждого треда
    size_prec.precalculate_shifts();
    // получаем размеры словарей
    DictsSize dict_sizes = size_prec.get_dicts_sizes();

    // устанавливаем полученные размеры
    for (auto it=dicts.begin(); it!=dicts.end(); it++) {
        it->second.resize(dict_sizes[it->first]);

    }
    INFO_LOG << "Index::LoadParallel dicts storage resized (mem: "
           << NMemInfo::GetMemInfo().VMS << ")";


    // сжимаем данные параллельно
    CompressingProcessor compressor(*this, size_prec, NUM_THREADS, BUFFER_SIZE);
    compressor.process_file(file_name);

    INFO_LOG << "data compressed in " << timer.Get().Seconds() << " sec,"
           << " mem(" << NMemInfo::GetMemInfo().VMS << ")";

    /* сортируем ключи каждого словаря */
    timer.Reset();

    for(auto it=dicts.begin(); it!=dicts.end(); it++) {
        std::sort(it->second.begin(), it->second.end(), _CompareKeys);
    }

    INFO_LOG << "dictionaries keys are sorted in " << timer.Get().Seconds() << " sec";

    INFO_LOG << "Index::Load finished (mem: " << NMemInfo::GetMemInfo().VMS << ")";
    return true;
}


bool Index::InitDictsParallel(const char* file_name, SizePrecalculator& size_prec) {
    INFO_LOG << "Index::InitDictsParallel started (mem: " << NMemInfo::GetMemInfo().VMS << ")";

    // вычисление размеров словарей
    size_prec.process_file(file_name);

    size_t full_size = size_prec.get_full_size();
    DictsSize dict_sizes = size_prec.get_dicts_sizes();

    // резервирование памяти
    INFO_LOG << " " << dict_sizes.size() << " dicts totally";
    INFO_LOG << " data image is " << full_size << " bytes";
    for(auto it = dict_sizes.begin(); it != dict_sizes.end(); it++) {
        dicts[it->first].reserve(it->second);
        INFO_LOG << " " << it->first << " : " << it->second << " items,"
               << " mem after reserve: " << NMemInfo::GetMemInfo().VMS;
    }

    INFO_LOG << "memory usage before full_data.resize(): " << NMemInfo::GetMemInfo().VMS;
    full_data.resize(full_size);
    INFO_LOG << "memory usage after full_data.resize(): " << NMemInfo::GetMemInfo().VMS;

    INFO_LOG << "Index::InitDicts finished (mem: " << NMemInfo::GetMemInfo().VMS << ")";
    return true;
}


bool Index::InitDicts(const char* file_name) {
    std::ifstream stream(file_name);

    if(stream.fail()) {
        return false;
    }

    INFO_LOG << "Index::InitDicts started (mem: " << NMemInfo::GetMemInfo().VMS << ")";

    // вычисление размеров словарей
    const unsigned BUFFER_SIZE = 1024 * 1024;
    char buffer[BUFFER_SIZE];
    size_t full_size = 1;
    std::map<std::string, unsigned> sizes;
    while(stream.good()) {
        stream.getline(buffer, BUFFER_SIZE);

        if(buffer[0]) {
            // название словаря
            char *dict_name = buffer;

            // ключ
            char *key = std::strchr(dict_name, '\t');
            if(!key) {
                INFO_LOG << "no key in line " << dict_name;
                continue;
            }
            *key = 0;
            key++;

            // значение
            char *value = std::strchr(key, '\t');
            if(!value) {
                INFO_LOG << "no value in line " << dict_name << '\t' << key;
                continue;
            }
            *value = 0;
            value++;

            sizes[dict_name] += 1;

            // свойства текущего словаря
            DictsProperties::const_iterator it = dicts_properties.find(dict_name);
            if(it == dicts_properties.end()) {
                DictProperties dp;
                dp.key_type = IKT_COMPRESSED;
                dp.value_type = IVT_COMPRESSED;
                it = dicts_properties.insert(DictsProperties::value_type(dict_name, dp)).first;
            }

            // размер пары "ключ-значение" в сжатых данных
            full_size +=
                sizeof(CompressedSize) + // длина ключа
                GetKeySize(key, it->second.key_type) + // ключ
                GetValueSize(value, it->second.value_type); // значение
        }
    }

    // резервирование памяти
    TMap<std::string, unsigned>::iterator it;
    INFO_LOG << " " << sizes.size() << " dicts totally";
    INFO_LOG << " data image is " << full_size << " bytes";
    for(it = sizes.begin(); it != sizes.end(); it++) {
        dicts[it->first].reserve(it->second);
        INFO_LOG << " " << it->first << " : " << it->second << " items";
    }

    INFO_LOG << "memory usage before full_data.resize(): " << NMemInfo::GetMemInfo().VMS;
    full_data.resize(full_size);
    INFO_LOG << "Index::InitDicts finished (mem: " << NMemInfo::GetMemInfo().VMS << ")";

    return true;
}

const unsigned char* Index::FindValue(const char* dict_name, const char* key, IndexValueType& value_type) const {
    if(is_debug) {
        std::cout << "incoming: " << dict_name << " / " << key << std::endl;
    }

    // словарь
    DictSet::const_iterator dict_it = dicts.find(dict_name);
    if(dict_it == dicts.end()) {
        return 0;
    }

    // тип ключа
    IndexKeyType key_type = IKT_COMPRESSED;
    DictsProperties::const_iterator dp_it = dicts_properties.find(dict_name);
    if(dp_it != dicts_properties.end()) {
        key_type = dp_it->second.key_type;
    }

    // кодирование ключа
    const unsigned BUFFER_SIZE = 1024;
    unsigned char buffer[BUFFER_SIZE];
    unsigned key_len = CompressKey(key, buffer + sizeof(CompressedSize), buffer + BUFFER_SIZE - 1, key_type);
    if(!key_len || key_len > MAX_COMPRESSED_SIZE) {
        return 0;
    }
    *((CompressedSize*)buffer) = (CompressedSize) key_len;

    if(is_debug) {
        std::cout << " encoded key: ";
        for(unsigned i = 0; i < key_len; i++) {
            for(unsigned j = 0; j < 8; j++) {
                std::cout << (((*(buffer + sizeof(CompressedSize) + i)) >> j) & 1);
            }
            std::cout << " ";
        }
        std::cout << std::endl;
    }

    // поиск ключа
    Dict::const_iterator value_it = std::lower_bound(dict_it->second.begin(), dict_it->second.end(), buffer, _CompareKeys);
    if(value_it == dict_it->second.end() || _CompareKeys(buffer, *value_it)) {
        return 0;
    }

    value_type = (dp_it != dicts_properties.end() ? dp_it->second.value_type : IVT_COMPRESSED);

    return *value_it;
}

TString Index::GetKey(const unsigned char* key_ptr, IndexKeyType key_type) const {
    if (key_type == IKT_COMPRESSED) {
        return huffman.Decompress(key_ptr);
    } else if (key_type == IKT_UINT32) {
        return ToString(*((uint32_t*) key_ptr));
    }
    return TString();
}

TString Index::GetValue(const unsigned char *value_ptr, IndexValueType value_type) const {
    CompressedSize value_len = *((CompressedSize*) value_ptr);

    if(is_debug) {
        std::cout << " found key:";
        for(unsigned i = 0; i < value_len; i++) {
            for(unsigned j = 0; j < 8; j++) {
                std::cout << (((*(value_ptr + sizeof(CompressedSize) + i)) >> j) & 1);
            }
            std::cout << " ";
        }
        std::cout << std::endl;
    }

    value_ptr += sizeof(CompressedSize) + value_len;

    TString result;
    if(value_type == IVT_COMPRESSED) {
        result = huffman.Decompress(value_ptr);
    } else if(value_type == IVT_UINT32V) {
        uint32_t *ints = (uint32_t*) value_ptr;
        size_t value_index = 0;

        if(is_debug) {
            std::cout << " # of values: " << ints[0] << std::endl;
        }

        TStringBuilder res;
        for(value_index = 0; value_index < ints[0]; value_index++) {
            if(res) {
                res << ' ';
            }

            res << ints[1 + value_index];
        }
        result = res;
    } else if(value_type == IVT_UINT32) {
        result = ToString(*((uint32_t*) value_ptr));
    }
    if(is_debug) {
        std::cout << "value: " << result << std::endl;
    }
    return result;
}

bool Index::SaveBinary(const char *file_name, int version) const {
    std::ofstream stream(file_name, std::ios::binary);
    return SaveBinary(stream, version);
}

bool Index::SaveBinary(std::ostream& stream, int version) const {
    if(stream.fail()) {
        return false;
    }

    // заголовок
    IndexFileHeader header;
    std::strcpy(header.signature, "INDEX");
    header.version = (char) version;
    header.num_dicts = dicts.size();
    header.data_size = full_data.size();
    stream.write((const char*) &header, sizeof(header));

    // сжатые данные
    stream.write((const char*) &full_data[0], full_data.size());

    // словари
    DictSet::const_iterator dict_it;
    for(dict_it = dicts.begin(); dict_it != dicts.end(); dict_it++) {
        // свойства словаря
        DictProperties properties = dicts_properties.find(dict_it->first)->second;

        // заголовок
        if(version == 0) {
            IndexFileDictV0 dict_info;
            dict_info.name_length = dict_it->first.size() + 1;
            dict_info.num_items = dict_it->second.size();
            stream.write((const char*) &dict_info, sizeof(dict_info));
        } else if(version == 1) {
            IndexFileDictV1 dict_info;
            dict_info.name_length = dict_it->first.size() + 1;
            dict_info.num_items = dict_it->second.size();
            dict_info.properties.key_type = properties.key_type;
            stream.write((const char*) &dict_info, sizeof(dict_info));
        } else if(version == 2) {
            IndexFileDict dict_info;
            dict_info.name_length = dict_it->first.size() + 1;
            dict_info.num_items = dict_it->second.size();
            dict_info.properties = properties;
            stream.write((const char*) &dict_info, sizeof(dict_info));
        }

        // название
        stream.write(dict_it->first.c_str(), dict_it->first.size() + 1);

        // указатели, преобразованные в смещения
        for(size_t i = 0; i < dict_it->second.size(); i++) {
            size_t shift = dict_it->second[i] - &full_data[0];
            stream.write((const char*) &shift, sizeof(shift));
        }
    }

    return true;
}

bool Index::LoadBinary(const char *file_name) {
    std::ifstream stream(file_name, std::ios::binary);
    return LoadBinary(stream);
}

bool Index::LoadBinary(std::istream& stream) {
    if(stream.fail()) {
        return false;
    }

    // заголовок
    IndexFileHeader header;
    stream.read((char*) &header, sizeof(header));

    if(stream.fail() || std::strcmp(header.signature, "INDEX")) {
        return false;
    }

    if(header.version < 0 || header.version > 2) {
        return false;
    }

    // сжатые данные
    full_data.resize(header.data_size);
    stream.read((char*) &full_data[0], full_data.size());

    // словари
    for(unsigned i = 0; i < header.num_dicts; i++) {
        // заголовок
        IndexFileDict dict_info;

        if(header.version == 0) {
            IndexFileDictV0 dict_info_v0;
            stream.read((char*) &dict_info_v0, sizeof(dict_info_v0));
            dict_info.name_length           = dict_info_v0.name_length;
            dict_info.num_items             = dict_info_v0.num_items;
            dict_info.properties.key_type   = IKT_COMPRESSED;
            dict_info.properties.value_type = IVT_COMPRESSED;
        } else if(header.version == 1) {
            IndexFileDictV1 dict_info_v1;
            stream.read((char*) &dict_info_v1, sizeof(dict_info_v1));
            dict_info.name_length           = dict_info_v1.name_length;
            dict_info.num_items             = dict_info_v1.num_items;
            dict_info.properties.key_type   = dict_info_v1.properties.key_type;
            dict_info.properties.value_type = IVT_COMPRESSED;
        } else {
            stream.read((char*) &dict_info, sizeof(dict_info));
        }

        // название словаря
        std::vector<char> buffer(dict_info.name_length);
        stream.read(&buffer[0], dict_info.name_length);
        TString name(&buffer[0]);
        dicts[name].resize(dict_info.num_items);
        dicts_properties[name] = dict_info.properties;

        // указатели
        for(size_t i = 0; i < dict_info.num_items; i++) {
            size_t shift;
            stream.read((char*) &shift, sizeof(shift));
            dicts[name][i] = &full_data[shift];
        }
    }

    return true;
}

void Index::DumpStats() const {
    INFO_LOG << "full data size: " << full_data.size();

    for(DictSet::const_iterator it = dicts.begin(); it != dicts.end(); ++it) {
        INFO_LOG<< it->first << " index size: " << it->second.size() * sizeof(Dict::value_type);

        const unsigned char *min_ptr = *std::min_element(it->second.begin(), it->second.end());
        const unsigned char *max_ptr = *std::max_element(it->second.begin(), it->second.end());
        INFO_LOG << it->first << " data size: " << (max_ptr - min_ptr);
    }
}

unsigned Index::GetKeySize(const TStringBuf& key, IndexKeyType type) const {
    if(type == IKT_UINT32) {
        return sizeof(uint32_t);
    }

    return huffman.CompressSize(key);
}

unsigned Index::GetValueSize(const TStringBuf& value, IndexValueType type) const {
    if(type == IVT_UINT32V) {
        unsigned value_count = 2;

        for (const auto& ch: value){
            if(ch == ' ') {
                value_count++;
            }
        }

        return value_count * sizeof(uint32_t);
    } else if(type == IVT_UINT32) {
        return sizeof(uint32_t);
    }

    return huffman.CompressSize(value);
}

unsigned Index::CompressKey(const TStringBuf& data, unsigned char* output, unsigned char* end, IndexKeyType type) const {
    if(type == IKT_UINT32) {
        *((uint32_t*) output) = FromString<uint32_t>(data);
        return sizeof(uint32_t);
    }

    return huffman.Compress(data, output, end);
}

unsigned Index::CompressValue(const TStringBuf& data, unsigned char *output, unsigned char *end, IndexValueType type) const {
    if(type == IVT_UINT32V) {
        uint32_t *value_count = (uint32_t*) output;
        uint32_t *uint_output = value_count + 1;
        const char *value_begin = data.Data();
        char *value_end;
        bool is_done = false;

        *value_count = 0;
        while(!is_done) {
            uint32_t value = std::strtoul(value_begin, &value_end, 10);

            if(value_end != value_begin) {
                (*value_count)++;
                *uint_output = value;
                uint_output++;
                value_begin = value_end;
            } else {
                is_done = true;
            }
        }

        return ((unsigned char*) uint_output) - output;
    } else if(type == IVT_UINT32) {
        char *value_end;
        *((uint32_t*) output) = std::strtoul(data.Data(), &value_end, 10);
        return sizeof(uint32_t);
    }

    return huffman.Compress(data, output, end);
}


////////////////////////////////////////
/* Методы класса CompressingProcessor */
////////////////////////////////////////

CompressingProcessor::CompressingProcessor(
        Index& index,
        SizePrecalculator& pos_shifts,
        const size_t num_threads,
        const size_t buffer_size)
    :FileProcessor(num_threads, buffer_size),
     _index(index),
     _pos_shifts(pos_shifts) {}

void CompressingProcessor::process_line(char* line, const size_t worker_i) {
    // получаем ссылки на объекты
    Index& index = _index;
    SizePrecalculator& shifts = _pos_shifts;

    // получаем указатель на место для записи результата текущего треда
    unsigned char* init_ptr = &(index.full_data[shifts.size_shifts[worker_i]]);
    unsigned char* curr_ptr = init_ptr;

    // получаем указатель на конец массива с результатами
    static unsigned char* end_ptr = &(index.full_data.back());

    // разбиваем строку на куски по табам
    TVector<TStringBuf> data;
    data.reserve(3);
    Split(line, "\t", data);
    if(data.size() != 3) return;
    TString dictName = TString(data[0]);

    // получаем свойства текущего словаря
    DictProperties dict_prop = index.dicts_properties[dictName];

    // сжимаем ключ и получаем длину результата
    unsigned key_len = index.CompressKey(
        data[1],
        curr_ptr + sizeof(CompressedSize),
        end_ptr,
        dict_prop.key_type);

    // сжимаем значение
    if (key_len <= MAX_COMPRESSED_SIZE) {
        // записываем в соотв. словарь указатель на начало текущего элемента
        // и сразу сдвигаем указатель для текущего треда на запись результата в словарь
        index.dicts[dictName][ shifts.dict_shifts[worker_i][dictName]++ ] = init_ptr;

        // пишем длину ключа в начало результата
        *((CompressedSize*) curr_ptr) = (CompressedSize)key_len;
        // сдвигаем указатель curr_ptr на ключ и его длину
        curr_ptr += key_len + sizeof(CompressedSize);

        // пишем сжатое значение в curr_ptr
        curr_ptr += index.CompressValue(
            data[2],
            curr_ptr,
            end_ptr,
            dict_prop.value_type);

        // увеличиваем сдвиг (для записи результата треда)
        // на размер только что обработанного элемента словаря
        shifts.size_shifts[worker_i] += curr_ptr - init_ptr;
    }
}

/////////////////////////////////////
/* Методы класса SizePrecalculator */
/////////////////////////////////////

SizePrecalculator::SizePrecalculator(
        const Index& index,
        const size_t num_threads,
        const size_t buffer_size)
    :FileProcessor(num_threads, buffer_size),
     _index(index) {
    // заводим структуры для результатов по каждому треду
    dict_sizes.resize(_num_threads);
    dicts_properties.resize(_num_threads);
    full_size.resize(_num_threads);
}

void SizePrecalculator::process_line(char* line, const size_t thr_i) {
    // разбиваем строку на куски по табам
    TVector<TStringBuf> data;
    data.reserve(3);
    Split(TStringBuf(line, strlen(line)), "\t", data);
    if(data.size() != 3) return;

    // увеличиваем размер словаря
    std::string str(data[0].data(), data[0].size());
    dict_sizes[thr_i][str]++;

    // получаем свойства текущего словаря
    auto it = dicts_properties[thr_i].find(data[0]);
    if(it == dicts_properties[thr_i].end()) {
        // если словарь встречается первый раз,
        // то заносим дефолтные данные по нему
        DictProperties dict_prop;
        dict_prop.key_type   = IKT_COMPRESSED;
        dict_prop.value_type = IVT_COMPRESSED;
        // сохраняем свойства и получаем итератор на позицию сохр. рез.
        it = dicts_properties[thr_i].insert(
            DictsProperties::value_type(data[0], dict_prop)
        ).first;
    }

    // пробавляем размер сжатой пары "ключ-значение" и длину ключа
    // к размеру данных текущего среда
    full_size[thr_i] +=
        sizeof(CompressedSize) + // длина ключа
        _index.GetKeySize(data[1], it->second.key_type) + // ключ
        _index.GetValueSize(data[2], it->second.value_type); // значение
}

DictsProperties SizePrecalculator::get_dict_properties() {
    // merge данных со свойствами словарей по всем тредам
    DictsProperties dp;
    for(size_t i=0; i<dicts_properties.size(); ++i)
        for(auto it=dicts_properties[i].begin(); it!=dicts_properties[i].end(); it++)
            // если нет данных по текущему словарю, то заносим их в результат
            if (dp.find(it->first) == dp.end()) dp.insert(*it);
    return dp;
}

void SizePrecalculator::precalculate_shifts() {
    // готовим структуры для записи сдвигов по каждому треду
    dict_shifts.clear();
    dict_shifts.resize(_num_threads);
    size_shifts.clear();
    size_shifts.resize(_num_threads);

    // получаем свойства всех словарей
    DictsProperties dict_prop = get_dict_properties();

    // будем писать данные по словарям
    // в лексикографическом порядке по их названиям
    // для нормализации формата

    // получаем названия словарей
    std::vector<std::string> dict_names;
    for(auto it=dict_prop.begin(); it!=dict_prop.end(); it++)
        dict_names.push_back(it->first);

    // сортируем названия в лексикографическом порядке
    std::sort(dict_names.begin(), dict_names.end());

    // подсчитываем сдвиги для каждого треда
    for(size_t thr_i=0; thr_i<_num_threads; ++thr_i) {
        for(std::string& name: dict_names) {
            // сдвиг для записи результата по словарю name для треда thr_i
            dict_shifts[thr_i][name] = thr_i>0 ?
                dict_shifts[thr_i-1][name] + dict_sizes[thr_i-1][name] : 0;
        }
        // сдвиг для записи сжатых данных для треда thr_i
        size_shifts[thr_i] = thr_i>0 ?
            size_shifts[thr_i-1] + full_size[thr_i-1]: 0;
    }
}

size_t SizePrecalculator::get_full_size() {
    // сумма по размеру данных в каждом треде + 1
    return std::accumulate(full_size.begin(), full_size.end(), (size_t)0) + 1;
}

DictsSize SizePrecalculator::get_dicts_sizes() {
    DictsSize sizes;
    // суммируем размеры словарей в каждом треде
    for(size_t thr_i=0; thr_i<_num_threads; ++thr_i)
        for(auto it=dict_sizes[thr_i].begin(); it!=dict_sizes[thr_i].end(); it++)
            sizes[it->first] += it->second;
    return sizes;
}

void SizePrecalculator::set_dicts_properties(const DictsProperties& props) {
    // удаляем старые данные
    dicts_properties.clear();

    // заносим данные со свойствами словарей в структуру для каждого треда
    dicts_properties.resize(_num_threads, props);
}

void SizePrecalculator::_clear_data(const size_t thr_i) {
    // чистим данные для треда thr_i
    dict_sizes[thr_i].clear();
    full_size[thr_i] = 0;
    // dicts_properties может быть предварительно установлен, его не чистим
}
