#pragma once

#include <rt-research/broadmatching/scripts/cpp-source/common/HuffmanTree.h>

#include <util/string/builder.h>

#include <iostream>
#include <fstream>
#include <string>
#include <cstring>
#include <numeric>
#include <algorithm>

#include <vector>
#include <map>
#include <unordered_map>

typedef std::map<std::string, size_t> DictsSize;
typedef std::vector<const unsigned char*> Dict;
typedef std::map<std::string, Dict> DictSet;

struct IndexFileHeader;
class  CompressingProcessor;
class  SizePrecalculator;

class Index {
public:

    Index(bool is_debug, const HuffmanTree& tree);

    // свойства словарей
    void        SetDictsProperties(const DictsProperties& dp)   { dicts_properties = dp; }

    // загрузка словарей
    bool        Load(const char* file_name);
    // паралелльная загрузка словарей
    bool        LoadParallel(const char* file_name);

    // найти адрес значения по ключу
    // если пары "словарь-ключ" не существует, возвращает 0
    const unsigned char*    FindValue(const char* dict_name, const char* key, IndexValueType& value_type) const;

    // получить значение по указанному адресу
    TString       GetKey(const unsigned char* value_ptr, IndexKeyType key_type) const;
    TString       GetValue(const unsigned char *value_ptr, IndexValueType value_type) const;

    unsigned    NumDicts() const    { return dicts.size(); }

    // загрузка/сохранение бинарного файла
    bool        SaveBinary(std::ostream& stream, int version) const;
    bool        LoadBinary(std::istream& stream);
    bool        SaveBinary(const char *file_name, int version) const;
    bool        LoadBinary(const char *file_name);

    // статистика
    void        DumpStats() const;

private:

    // работа с ключами и значениями разных типов
    unsigned    GetKeySize(const TStringBuf& key, IndexKeyType type) const;
    unsigned    GetValueSize(const TStringBuf& key, IndexValueType type) const;
    unsigned    CompressKey(const TStringBuf&, unsigned char* output, unsigned char* end, IndexKeyType type) const;
    unsigned    CompressValue(const TStringBuf& data, unsigned char* output, unsigned char* end, IndexValueType type) const;

    // предварительное вычисление размеров словарей
    bool        InitDicts(const char* file_name);
    // параллельное предварительное вычисление размеров словарей
    bool        InitDictsParallel(const char* file_name, SizePrecalculator& size_prec);

private:
    bool is_debug;
    const HuffmanTree&          huffman;
    DictSet                     dicts;
    DictsProperties             dicts_properties;
    std::vector<unsigned char>  full_data;

    friend CompressingProcessor;
    friend SizePrecalculator;
};

/* Класс для параллельного сжатия cdict'а

  Проходится по файлу в n тредов и сразу пишет сжатые данные в index.
  В конструкторе передается указатель на отработавший SizePrecalculator.

  Параллелизация по данным. Треды пишут сразу на нужное место, подсчитанное в SizePrecalculator.

  NOTICE: для оптимизации во время работы изменяется SizePrecalculator.
*/

class CompressingProcessor: public FileProcessor {
private:
    // указатель на Index, в который будем писать данные
    Index& _index;
    // предподсчитанные смещения для записи результата каждого треда
    SizePrecalculator& _pos_shifts;

    void _clear_data(const size_t ) {};

public:
    CompressingProcessor(
            Index& index,
            SizePrecalculator& pos_shifts,
            const size_t num_threads,
            const size_t buffer_size);

    void process_line(char* line, const size_t thr_i);

    friend Index;
};

/* Класс для предподсчёта размеров словарей и смещений для записи
 результата в каждом треде.

 Проходится в n тредов по файлу и вычисляет:
   1) dict_sizes       : размеры словарей
   2) dicts_properties : свойства словарей
   3) dict_shifts      : смещения по тредам для записи рез. по словарям
   4) size_shifts      : смещения по тредам для записи рез. сжатия
*/

class SizePrecalculator: public FileProcessor {
private:
    // индекс, для которого жмём данные
    const Index& _index;

    // размер данных по тредам
    std::vector<size_t> full_size;

    // размеры словарей по тредам
    TVector<DictsSize>       dict_sizes;
    // свойства словарей по тредам
    TVector<DictsProperties> dicts_properties;

    // смещения по тредам для записи рез. по словарям
    std::vector<DictsSize> dict_shifts;
    // смещения по тредам для записи рез. сжатия
    std::vector<size_t>    size_shifts;

    void _clear_data(const size_t thr_i);

public:
    // конструктор
    SizePrecalculator(
            const Index& index,
            const size_t num_threads,
            const size_t buffer_size);

    // обработка строки - подсчитывает итоговые размеры
    void process_line(char* line, const size_t thr_i);

    // подсчитывает итоговые смещение на основе итоговых размеров
    // !вызывать имеет смысл только после process_file - нужны итоговые размеры
    // результат заносит в dict_shifts и size_shifts
    void precalculate_shifts();

    // установка свойств словарей извне
    void set_dicts_properties(const DictsProperties& props);

    // получить итоговый размер всех сжатых данных
    size_t get_full_size();
    // получить итоговые свойства всех словарей
    DictsProperties get_dict_properties();
    // получить кол-во записей в каждом словаре
    DictsSize get_dicts_sizes();

    friend Index;
    friend CompressingProcessor;
};

