#pragma once


#include <cstring>
#include <thread>
#include <fstream>
#include <vector>

/* Абстрактные класс для параллельной обработки строк файла на std::thread'ах

  Для конкретного случая использования наследуемся от этого класса и
определяем методы 1) _clear_data и 2) process_line, затем вызываем process_file
  Метод process_file может изменять переменные класса (записывать результат,
статистики и тп).
  Кол-во тредов и размера буфера для одной строки указывается в конструкторе.

-------------------------------------------------------------------------------
  Пример использования можно посмотреть в HuffmatTree.h или cdict/Index.h
*/

class FileProcessor {
protected:
    const size_t _buffer_size;
    const size_t _num_threads;

    // функция, которая будет вызываться при запуске треда
    virtual void _clear_data(const size_t worker_i) = 0;
public:
    // конструктор
    FileProcessor(const size_t num_threads, const size_t buffer_size):
        _buffer_size(buffer_size), _num_threads(num_threads) {}

    // метод, который создает треды,
    // запускает каждый и ждёт окончания работы всех
    bool process_file(const char* file_path);

    // метод, который запускается на каждом треде
    // считывает построчно данные из своей чанки и передает их в process_line
    void worker(
        const char *file_path,
        const size_t worker_i,
        const size_t chunk_size);

    // обработчик одной строки файла, запускается на треде
    // получает строку и свой номер
    virtual void process_line(char* line, const size_t worker_i) = 0;

    static size_t get_file_size(const char *file_path);
};

