#pragma once

#include <rt-research/broadmatching/scripts/cpp-source/common/FileProcessor.h>

#include <util/generic/map.h>
#include <util/generic/string.h>
#include <util/generic/vector.h>

#include <vector>
#include <list>
#include <string>

#define HUFFMAN_END_MARKER "\\end"

typedef unsigned long long WordBits;

// тип данных для хранения ключей
enum IndexKeyType {
    IKT_COMPRESSED  = 0,
    IKT_UINT32      = 1
};

// тип данных для хранения значений
enum IndexValueType {
    IVT_COMPRESSED  = 0,
    IVT_UINT32V     = 1,
    IVT_UINT32     = 2
};

struct DictPropertiesV1 {
    IndexKeyType    key_type;
};

struct DictProperties {
    IndexKeyType    key_type;
    IndexValueType  value_type;
};

struct DictFileHeader {
    char    signature[5];   // "DICT"
    char    version;        // 0
    size_t  nodes_block_size;
    size_t  num_words;
};


typedef TMap<TString, DictProperties> DictsProperties;
typedef TMap<TString, size_t> DictFrequencies;

class HuffmanTree {
public:

    // генерация по словарю частот
    void Generate(const DictFrequencies& freqs);

    // генерация по входному файлу
    bool GenerateFromFile(const char *file_name, const DictsProperties& dp);

    // генерация по входному файлу с параллельным подсчетом частот
    bool GenerateFromFileParallel(const char *file_name, const DictsProperties& dict_prop);

    // загрузка/сохранение в файл
    bool        Load(std::istream& stream, bool verbose = true);
    bool        Load(const TStringBuf& file_name, bool verbose = true);
    bool        Save(const TStringBuf& file_name) const;
    bool        Save(std::ostream& stream) const;

    // сжатие
    unsigned        CompressSize(const TStringBuf& data) const;
    unsigned        Compress(const TStringBuf& data, unsigned char* output, unsigned char* end) const;
    TString        Decompress(const unsigned char* code) const;
    bool Encode(
        const std::vector<std::string>& data,
        std::vector<char>& encoded_data,
        bool add_end_marker = false) const;

    std::string Decode(
        const std::vector<char>& encoded_data,
        char words_delim) const;

public:

    // узел дерева
    struct Node {
        Node() : freq(0), parent(0), left(0), right(0)  {}

        std::string     text;
        size_t          freq;
        Node            *parent, *left, *right;
    };

    // кодирование слова
    struct WordCode {
        unsigned char*  Write(unsigned char* output, unsigned char* end, unsigned& bit_count) const;
        void Write(std::vector<char>& dst, unsigned& bit_count) const;

        const char*     text;
        WordBits        bits;
        unsigned char   num_bits;
    };

private:

    const WordCode* FindWord(const char* data) const;

    // упаковка дерева
    void        PackTree();
    size_t      PackNode(const Node* node, size_t shift);

    // построение обратного словаря
    void        BuildWordCodes();
    void        BuildWordCodes(size_t shift, WordBits bits, unsigned char num_bits);

private:

    // неупакованное дерево
    std::list<Node> nodes;
    Node*           root_node;

    // упакованное дерево
    std::vector<char>   packed_nodes;
    unsigned            num_leaf_nodes;

    // обратный словарь (слово -> код)
    std::vector<WordCode>   word_codes;
    WordCode*               wc_end_marker;
};


/* Класс для параллельного WordCount на тредах */

class WordCountProcessor: public FileProcessor {
private:
    // словарь с частотами слов для каждого треда
    std::vector<DictFrequencies> _worker_df;

    // свойства словарей
    const DictsProperties* _dict_prop;

    void _clear_data(const size_t worker_i);

public:
    // конструктор, принимает
    WordCountProcessor(
        const DictsProperties* dict_prop, // указатель на свойства словаря
        const size_t num_threads,         // кол-во тредов
        const size_t buffer_size);        // размер буффера для входной строки

    // обработчик одной строки
    void process_line(char* line, const size_t worker_i);

    // merge частот слов по тредам в один словарь
    DictFrequencies join_results();
};
