#include "Server.h"

#include "Session.h"

#include <library/cpp/logger/global/global.h>

#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <iostream>

#include <arpa/inet.h>
#include <fcntl.h>
#include <netinet/in.h>
#include <sys/types.h>
#include <unistd.h>

#define USE_IPV6    1

Server::Server(int port){
#if USE_IPV6
    sockaddr_in6 addr;
#else
    sockaddr_in addr;
#endif

    if(pthread_mutex_init(&mutex_response, 0) != 0) {
        ERROR_LOG << "can't create mutex";
        exit(1);
    }

    int one = 1;

    // создаём сокет
#if USE_IPV6
    socket_fd = socket(PF_INET6, SOCK_STREAM, 0);
#else
    socket_fd = socket(PF_INET, SOCK_STREAM, 0);
#endif

    if(socket_fd == -1) {
        ERROR_LOG << "can't create socket";
        exit(1);
    }

    // настройки сокета
    int flags = fcntl(socket_fd, F_GETFL);
    fcntl(socket_fd, F_SETFL, flags | O_NONBLOCK);
    if(setsockopt(socket_fd, SOL_SOCKET, SO_REUSEADDR, &one, sizeof(one)) == -1 ||
#ifdef USE_SO_NOSIGPIPE
        setsockopt(socket_fd, SOL_SOCKET, SO_NOSIGPIPE, &one, sizeof(one)) == -1 ||
#endif
        setsockopt(socket_fd, SOL_SOCKET, SO_KEEPALIVE, &one, sizeof(one)) == -1) {
        ERROR_LOG << "setsockopt() failed";
        exit(1);
    }

    // привязываем сокет
#if USE_IPV6
    int zero = 0;
    if(setsockopt(socket_fd, IPPROTO_IPV6, IPV6_V6ONLY, &zero, sizeof(zero)) == -1) {
        ERROR_LOG << "setsockopt(IPV6_V6ONLY) failed";
        exit(1);
    }
    addr.sin6_family = AF_INET6;
    addr.sin6_port = htons(port);
    addr.sin6_addr = in6addr_any;
#else
    addr.sin_family = AF_INET;
    addr.sin_port = htons(port);
    addr.sin_addr.s_addr = INADDR_ANY;
    memset(&(addr.sin_zero), 0, 8);
#endif

    if(bind(socket_fd, (sockaddr*)&addr, sizeof(addr)) == -1) {
        ERROR_LOG << "can't bind socket";
        exit(1);
    }
}

Server::~Server() {
    if(socket_fd != -1) {
        shutdown(socket_fd, SHUT_RDWR);
    }
}

struct ProcessSessionData {
    Server*     server;
    int         client_id;
    std::string client_addr;
};

void* FnProcessSession(void* arg) {
    // гарантируем очистку ресурсов
    pthread_detach(pthread_self());

    // создание сессии
    ProcessSessionData* data = (ProcessSessionData*) arg;
    Session session(data->client_id, data->server, data->client_addr);
    delete data;

    // обработка сессии
    session.Run();

    return 0;
}

void Server::Run() {
    if(listen(socket_fd, 20) == -1) {
        ERROR_LOG << "can't listen to the socket";
        return;
    }

    // цикл обработки входящих соединений
    INFO_LOG << "start listening...";
    is_stopped = false;
    while(!is_stopped) {
        fd_set fds;
        timeval timeout;
        int rc;

        timeout.tv_sec = 5;
        timeout.tv_usec = 0;
        FD_ZERO(&fds);
        FD_SET(socket_fd, &fds);
        rc = select(socket_fd + 1, &fds, 0, 0, &timeout);

        if(is_stopped) {
            break;
        }

        if(rc < 0) {
            ERROR_LOG << "select() before accept() failed";
//            is_stopped = true;
        } else if(rc > 0) {
#if USE_IPV6
            char straddr[INET6_ADDRSTRLEN];
            sockaddr_in6 addr;
#else
            sockaddr_in addr;
#endif
            socklen_t addr_size = sizeof(addr);
            int client_id = accept(socket_fd, (sockaddr*)&addr, &addr_size);

            if(client_id != -1) {
                pthread_t thr;
                ProcessSessionData *data = new ProcessSessionData;

#if USE_IPV6
                const std::string connection_from(inet_ntop(AF_INET6, &addr.sin6_addr, straddr, sizeof(straddr)));
#else
                const std::string connection_from(inet_ntoa(addr.sin_addr));
#endif
                INFO_LOG << "connection from " << connection_from << ", id=" << client_id;

                // создаём тред для обработки соединения
                data->server = this;
                data->client_id = client_id;
                data->client_addr = connection_from;

                if(pthread_create(&thr, 0, FnProcessSession, data)) {
                    ERROR_LOG << "can't create pthread";
                }
            } else {
                WARNING_LOG << "accept() failed";
            }
        }
    }
    INFO_LOG << "end listening";
}

void Server::LockResponse() {
    pthread_mutex_lock(&mutex_response);
}

void Server::UnlockResponse() {
    pthread_mutex_unlock(&mutex_response);
}

void Server::ExecCommand(const TStringBuf& /* command */, Session* session) {
    session->SendResponse("ERROR: ExecCommand() is not implemented");
}

void Server::Stop() {
    is_stopped = true;
}
