#include "Session.h"

#include "Server.h"

#include <library/cpp/logger/global/global.h>

#include <util/string/strip.h>

#include <cerrno>
#include <cstring>
#include <iostream>

#include <poll.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <unistd.h>

template <class It>
struct TChompSpaceAdapter {
    bool operator()(const It& it) const noexcept {
        return *it == '\r' || *it == '\n';
    }
};

template <class It>
TChompSpaceAdapter<It> ChompSpaceAdapter(It) {
    return {};
}

template <class T>
static inline bool ChompString(const T& from, T& to) {
    return StripString(from, to, ChompSpaceAdapter(from.begin()));
}

#define USE_POLL_SERVER

Session::Session(int client_id, Server* server) {
    this->client_id = client_id;
    this->server = server;
    this->client_addr = "not_set";
}

Session::Session(int client_id, Server* server, const std::string& client_addr) {
    this->client_id = client_id;
    this->server = server;
    this->client_addr = client_addr;
}

Session::~Session() {
    if(client_id) {
        INFO_LOG << "connection " << client_id << " is closed";
        shutdown(client_id, SHUT_RDWR);
        close(client_id);
    }
}

void* FnSessionThread(void* session) {
    ((Session*)session)->Run();

    return 0;
}

void Session::Run() {
    const unsigned BUFFER_SIZE = 128;
    const time_t TIMEOUT_SECONDS = 30;
    char buffer[BUFFER_SIZE];
    std::string cmd;
    int rc;

#ifdef USE_POLL_SERVER
    struct pollfd poll_set;
    poll_set.fd = client_id;
    poll_set.events = POLLIN;
    poll_set.revents = 0;
#else
    struct timeval timeout;
    timeout.tv_sec = TIMEOUT_SECONDS;
    timeout.tv_usec = 0;
#endif

    is_stopped = false;
    while(!is_stopped && !server->IsStopped()) {
#ifdef USE_POLL_SERVER
        rc = poll(&poll_set, 1, TIMEOUT_SECONDS * 1000);
#else
        fd_set fds;
        FD_SET(client_id, &fds);
        rc = select(client_id + 1, &fds, NULL, NULL, &timeout);
#endif

        if(rc < 0) {
#ifdef USE_POLL_SERVER
            INFO_LOG << "poll() for connection " << client_id << " failed (" << strerror(errno) << ")";
#else
            INFO_LOG << "select() for connection " << client_id << " failed";
#endif
            is_stopped = true;
        } else if(rc > 0) {
            int num_bytes = recv(client_id, buffer, BUFFER_SIZE - 1, 0);

            if(num_bytes <= 0) {
                is_stopped = true;
            } else {
                buffer[num_bytes] = 0;
                cmd += buffer;
            }

            if(cmd.size() && cmd.back() == '\n') {
                if(!server->IsStopped()) {
                    TString command(cmd);
                    ChompString(command, command);
                    server->ExecCommand(command, this);
                }

                cmd = "";
            } else if (is_stopped) {
                WARNING_LOG << "recv receives 0 bytes, looks like full message wasn't send";
            }
        } else {
            INFO_LOG << "no data from " << client_id << " for " << TIMEOUT_SECONDS << " seconds";
            is_stopped = true;
        }
    }
}

void Session::SendPartResponse(const TStringBuf& data) {
    if(client_id && send(client_id, data.Data(), data.Size(), MSG_NOSIGNAL) == -1) {
        is_stopped = true;
    }
}

void Session::SendResponse(const char* data) {
    TStringBuilder resp;
    resp << data << '\n';

    server->LockResponse();
    SendPartResponse(resp);
    server->UnlockResponse();
}

void Session::Stop() {
    is_stopped = true;
}

