#include "http_server.h"

#include <util/generic/string.h>
#include <util/string/builder.h>
#include <util/datetime/base.h>
#include <util/datetime/cputimer.h>
#include <util/stream/file.h>
#include <util/string/cast.h>

#include <library/cpp/json/json_reader.h>
#include <library/cpp/json/json_writer.h>
#include <library/cpp/http/misc/parsed_request.h>
#include <library/cpp/logger/global/global.h>


using namespace NIrt;
using namespace NJson;

inline static TLog& GetLogger() {
    return TLoggerOperator<TGlobalLog>::Log();
}

static TJsonValue GetErrorJson(const TString& errmsg, const TString& field = "errmsg") {
    TJsonValue response = TJsonValue(EJsonValueType::JSON_MAP);
    response[field] = errmsg;
    return response;
}

static TString TrimLongLine(TStringBuf line, const ui32 maxLength = 600) {
    static const TString placeholder = "(...)";

    if (line.size() <= maxLength) {
        return ToString(line);
    }

    return TStringBuilder() << line.SubStr(0, maxLength / 2) << placeholder << line.SubStr(line.size() - (maxLength / 2) + placeholder.size());
}

bool TRpcProxyHttpReplier::DoReply(const TReplyParams& params) {
    auto responsePair = DoReplyImpl(params);

    TStringStream responseBody;
    WriteJson(&responseBody, &responsePair.second);

    // todo: поддержать сжатие ответа
    THttpResponse fullResponse(responsePair.first);
    fullResponse.SetContent(responseBody.Str());
    params.Output << fullResponse;

    return true;
}

TRpcProxyHttpReplier::THttpResponseValue TRpcProxyHttpReplier::DoReplyImpl(const TReplyParams& params) {
    TParsedHttpFull req(params.Input.FirstLine());
    if (req.Method != "POST") {
        return { HttpCodes::HTTP_METHOD_NOT_ALLOWED, GetErrorJson("Use POST method") };
    }

    TJsonValue input;
    if (!ReadJsonTree(&params.Input, &input)) {
        return { HttpCodes::HTTP_BAD_REQUEST, GetErrorJson("Error while parsing input json") };
    }

    if (!input.IsMap() || !input.Has("action")) {
        return { HttpCodes::HTTP_BAD_REQUEST, GetErrorJson("Provide 'action'") };
    }

    const TString& action = input["action"].GetStringSafe();

    if (action == "ping") {
        GetLogger() << "Action 'ping': pong";
        return { HttpCodes::HTTP_OK, GetErrorJson("pong") };
    } else if (action == "shutdown") {
        return ActionShutdown(input);
    } else if (action == "reopen_log") {
        GetLogger() << "Action 'reopen_log': reopen...";
        GetLogger().ReopenLog();
        return { HttpCodes::HTTP_OK, GetErrorJson("OK") };
    } else if (action == "select") {
        return ActionSelect(input);
    } else {
        GetLogger() << "Action '" << action << "': unknown";
        return { HttpCodes::HTTP_I_AM_A_TEAPOT, GetErrorJson("Unknown action: '" + action + "'") };
    }
}

TRpcProxyHttpReplier::THttpResponseValue TRpcProxyHttpReplier::ActionShutdown(TJsonValue input) {
    Y_UNUSED(input);

    GetLogger() << "Action 'shutdown': shutting down...";
    Requester->Shutdown();
    HttpServ()->Shutdown();

    return { HttpCodes::HTTP_OK, GetErrorJson("Shutting down...") };
}

TRpcProxyHttpReplier::THttpResponseValue TRpcProxyHttpReplier::ActionSelect(TJsonValue input) {
    if (!input.Has("what") || !input.Has("from") || !input.Has("where")) {
        GetLogger() << "Action 'select': incomplete request";
        return { HttpCodes::HTTP_BAD_REQUEST, GetErrorJson("Action 'select' requires 'what', 'from' and 'where' parameters") };
    }

    const TString& what = input["what"].GetStringSafe();
    const TString& from = input["from"].GetStringSafe();
    const TString& where = input["where"].GetStringSafe();
    const TString& meta = input["meta"].GetString();
    const i64 limit = input["limit"].GetInteger();

    // todo: т.к. мы склеиваем what, from и where без санитайзинга, acl тут на правах не забора, а надписи - м.б. стоит поправить?
    if (!Acl->Allowed(from)) {
        GetLogger() << "Action 'select': request to a forbidden path '" << from << "'";
        return { HttpCodes::HTTP_FORBIDDEN, GetErrorJson("requested path is not whitelisted") };
    }

    TSimpleTimer timer;
    TStringBuilder queryBuilder;

    queryBuilder << what << " from [" << from << "] where " << where;
    if (limit > 0) {
        queryBuilder << " limit " << ToString(limit);
    }

    const TString query = queryBuilder;

    auto requesterResponse = Requester->DoSelect(query);
    auto rowCount = requesterResponse.Success ? requesterResponse.Data.GetRef().GetArray().size() : 0;
    auto elapsed = timer.Get().MilliSeconds();

    GetLogger() << "Action 'select': '" << TrimLongLine(query) << "', "
                << elapsed << " ms elapsed, "
                << "success: " << (requesterResponse.Success ? "yes" : "no") << ", "
                << "rows: " << rowCount << ", "
                << "meta: '" << meta << "'";

    if (requesterResponse.Success) {
        TJsonValue response = TJsonValue(EJsonValueType::JSON_MAP);
        response["data"] = requesterResponse.Data.GetRef();
        response["milliseconds_elapsed"] = elapsed;
        return { HttpCodes::HTTP_OK, response };
    } else {
        // todo: начать отличать ошибки классов 400х и 500х по кодам возврата YT
        return { HttpCodes::HTTP_INTERNAL_SERVER_ERROR, GetErrorJson(requesterResponse.Message.GetRef()) };
    }
}


