# coding=utf8
import argparse
import filelock
import logging
import os
import requests
import time

import yt.yson as yson

import bm.yt_tools
import irt.broadmatching.common_options


logging.basicConfig(format='%(asctime)s\t[%(process)d]\t%(message)s', level=logging.INFO)
logger = logging.getLogger(__name__)


def gen_config(dt_proxy_params):
    yt_config = bm.yt_tools.get_yt_bm_config()

    if 'token_path' in yt_config:
        with open(yt_config['token_path'], 'r') as tokenfile:
            yt_token = tokenfile.read().strip()
    elif 'token' in yt_config:
        yt_token = yt_config['token']
    else:
        raise ValueError('No token in yt_config')

    yt_params = yson.YsonMap()
    yt_params['user'] = 'robot-bm-admin'  # todo: не хардкодить
    yt_params['token'] = yt_token

    connection = yson.YsonMap()
    connection['cluster_url'] = os.environ['YT_PROXY']
    yt_params['connection'] = connection

    acl = yson.YsonList()
    for path in dt_proxy_params['proxy_acl']:
        acl.append(path)

    config = yson.YsonMap()
    config['yt_params'] = yt_params
    config['acl_params'] = acl
    config['proxy_port'] = dt_proxy_params['proxy_port']

    with open(dt_proxy_params['proxy_config'], 'w') as outfile:
        yson.dump(config, outfile, yson_format='pretty', indent=4)
        outfile.write('\n')


def run_proxy(dt_proxy_params):
    lock = filelock.FileLock('{}.pid'.format(dt_proxy_params['proxy_lockname']))

    try:
        with lock.acquire(timeout=0):
            logger.info('Generate config for cluster \'%s\'...', os.environ['YT_PROXY'])
            gen_config(dt_proxy_params)

            cmd = '{0} --config {1} --log {2}'.format(dt_proxy_params['proxy_binary'], dt_proxy_params['proxy_config'], dt_proxy_params['proxy_log'])
            logger.info('Config ready, run:')
            logger.info(cmd)
            os.system(cmd)
            logger.info('Done')
    except filelock.Timeout:
        logger.warning('Another instance of this application currently holds the lock.')


def get_local_proxy(dt_proxy_params):
    return 'http://127.0.0.1:{0}/'.format(dt_proxy_params['proxy_port'])


def post_action(dt_proxy_params, action):
    local_proxy = get_local_proxy(dt_proxy_params)
    r = requests.post(local_proxy, json={'action': action})
    logger.info('Result for action \'{0}\': \'{1}\''.format(action, r.text))


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument('-a', '--action')
    args = parser.parse_args()

    dt_proxy_params = irt.broadmatching.common_options.get_options()['dt_proxy_params']

    if args.action == 'start':
        run_proxy(dt_proxy_params)
    elif args.action == 'restart':
        try:
            post_action(dt_proxy_params, 'shutdown')
            time.sleep(3)
        except Exception as e:
            logger.exception('Looks like proxy is not running', e)
        finally:
            run_proxy(dt_proxy_params)
    elif args.action == 'shutdown' or args.action == 'reopen_log':
        post_action(dt_proxy_params, args.action)
    else:
        raise RuntimeError('Unknown action \'{0}\''.format(args.action))


if __name__ == '__main__':
    if 'YT_PROXY' not in os.environ:
        os.environ['YT_PROXY'] = 'hahn'
    main()
