#include "rpc_requester.h"
#include "http_server.h"

#include <irt/common/lib/logger/logger.h>

#include <util/string/builder.h>
#include <util/system/getpid.h>
#include <util/system/daemon.h>
#include <library/cpp/getopt/small/last_getopt.h>
#include <yt/yt/core/ytree/convert.h>
#include <yt/yt/core/misc/shutdown.h>

int RunProxy(const TString& configFileName, const TString& logFileName, bool daemonize)
{
    NIRT::InitLog<NIRT::TTabLoggerFormatter>({.ConsoleOnly=true});
    INFO_LOG << "Started with config '" << configFileName << "', writing log to '" << logFileName << "'";

    INFO_LOG << "Read config...";
    TIFStream stream(configFileName);
    auto configMap = NYT::NYTree::ConvertToNode(&stream)->AsMap();
    auto ytParams = configMap->GetChildOrThrow("yt_params");
    auto aclParams = configMap->GetChildOrThrow("acl_params");
    auto proxyPort = configMap->GetChildOrThrow("proxy_port")->AsInt64()->GetValue();

    if (daemonize) {
        INFO_LOG << "Daemonize...";
        NDaemonMaker::MakeMeDaemon(
            NDaemonMaker::closeStdIoOnly,
            NDaemonMaker::openNone,
            NDaemonMaker::chdirNone
        );
    }

    INFO_LOG << "Make rpc requester & acl checker...";
    auto requester = NIrt::TRpcProxyRequesterPtr(new NIrt::TRpcProxyRequester(ytParams));
    auto acl = NIrt::TRpcProxyAclPtr(new NIrt::TRpcProxyAcl(aclParams));

    INFO_LOG << "Make http server...";
    THttpServer::TOptions httpServerOptions;
    httpServerOptions.Port = proxyPort;
    httpServerOptions.KeepAliveEnabled = true;

    auto callback = THolder<THttpServer::ICallBack>(new NIrt::TRpcProxyHttpCallBack(requester, acl));
    THttpServer httpServer(callback.Get(), httpServerOptions);

    INFO_LOG << "Start http server...";
    httpServer.Start();

    INFO_LOG << "Wait for requests...";
    httpServer.Wait();

    INFO_LOG << "Done waiting.";
    return 0;
}

int main(int argc, const char** argv)
{
    TString configFileName;
    TString logFileName;
    bool daemonize;

    NLastGetopt::TOpts opts;
    opts.AddLongOption("config").Required().RequiredArgument().StoreResult(&configFileName);
    opts.AddLongOption("log").Optional().DefaultValue("/dev/stdout").StoreResult(&logFileName);
    opts.AddLongOption("daemonize").Optional().NoArgument().SetFlag(&daemonize);
    NLastGetopt::TOptsParseResult (&opts, argc, argv);

    int ret = RunProxy(configFileName, logFileName, daemonize);

    NYT::Shutdown();
    return ret;
}
