#include "rpc_requester.h"

#include <yt/yt/client/api/connection.h>
#include <yt/yt/client/api/client.h>
#include <yt/yt/client/api/transaction.h>
#include <yt/yt/client/api/rowset.h>

#include <yt/yt/client/api/rpc_proxy/config.h>
#include <yt/yt/client/api/rpc_proxy/connection.h>

#include <yt/yt/core/logging/log_manager.h>
#include <yt/yt/core/logging/config.h>

#include <yt/yt/core/net/address.h>
#include <yt/yt/core/net/dns_resolver.h>

#include <yt/yt/core/rpc/dispatcher.h>

#include <yt/yt/core/yson/writer.h>

#include <library/cpp/json/yson/json2yson.h>
#include <library/cpp/json/json_value.h>
#include <library/cpp/json/json_writer.h>

using namespace NIrt;
using namespace NJson;

using namespace NYT;
using namespace NYT::NApi;
using namespace NYT::NYTree;
using namespace NYT::NNet;
using namespace NYT::NApi::NRpcProxy;
using namespace NYT::NTableClient;

TRpcProxyRequester::TRpcProxyRequester(INodePtr ytParams) {
    NLogging::TLogManager::Get()->Configure(New<NLogging::TLogManagerConfig>());
    NNet::TAddressResolver::Get()->Configure(New<NNet::TAddressResolverConfig>());
    NRpc::TDispatcher::Get()->Configure(New<NRpc::TDispatcherConfig>());

    auto ytParamsMap = ytParams->AsMap();

    auto connectionConfig = New<NRpcProxy::TConnectionConfig>();
    connectionConfig->Load(ytParamsMap->GetChildOrThrow("connection"));
    auto connection = NRpcProxy::CreateConnection(connectionConfig);

    auto clientOptions = TClientOptions();
    clientOptions.User = ytParamsMap->GetChildOrThrow("user")->AsString()->GetValue();
    clientOptions.Token = ytParamsMap->GetChildOrThrow("token")->AsString()->GetValue();

    Client = connection->CreateClient(clientOptions);
}

void TRpcProxyRequester::Shutdown() {
    Client->Terminate();
}

TJsonValue UnversionedValueToNode(const TUnversionedValue& value) {
    switch (value.Type) {
        case EValueType::Int64:
            return TJsonValue(value.Data.Int64);
        case EValueType::Uint64:
            return TJsonValue(value.Data.Uint64);
        case EValueType::Double:
            return TJsonValue(value.Data.Double);
        case EValueType::Boolean:
            return TJsonValue(value.Data.Boolean);
        case EValueType::String:
            return TJsonValue(value.AsString());
        case EValueType::Null:
            return TJsonValue(EJsonValueType::JSON_NULL);
        case EValueType::Any: {
            TStringBuf in_buf(value.Data.String, value.Length);
            TJsonValue res;
            if (!NJson2Yson::DeserializeYsonAsJsonValue(in_buf, &res)) {
                ythrow yexception() << "Error in parse Yson ";
            }
            return res;
        }
        default:
            ythrow yexception() << "Unsupported value type " << ToString(value.Type);
    }
}

TJsonValue RowsetToJson(IUnversionedRowsetPtr rowset) {
    TJsonValue json(EJsonValueType::JSON_ARRAY);
    for (const auto& row : rowset->GetRows()) {
        TJsonValue rowJson(EJsonValueType::JSON_ARRAY);
        for (const auto& value : row) {
            rowJson.AppendValue(UnversionedValueToNode(value));
        }
        json.AppendValue(rowJson);
    }
    return json;
}

TRpcProxyResponse TRpcProxyRequester::DoSelect(const TString query) {
    auto result = Client->SelectRows(query).Get();

    TRpcProxyResponse resp;

    if (result.IsOK()) {
        resp.Success = true;
        resp.Data = RowsetToJson(result.Value().Rowset);
    } else {
        resp.Success = false;
        resp.Message = ToString(result);
    }

    return resp;
}

TRpcProxyAcl::TRpcProxyAcl(INodePtr aclParams) {
    for (const auto& node : aclParams->AsList()->GetChildren()) {
        AllowedPrefices.push_back(node->AsString()->GetValue());
    }
}

bool TRpcProxyAcl::Allowed(TString path) const {
    for (const auto& prefix : AllowedPrefices) {
        if (path.StartsWith(prefix)) {
            return true;
        }
    }
    return false;
}

