#include "SubphraserIndex.h"

#include <util/string/join.h>
#include <util/string/split.h>

const TStringBuf TERMINAL_NODE = " ";

class Node {
public:
    TMap<TString, Node> edges;
    TString data;
    bool is_terminal;

    Node()
    : is_terminal(false) {

    }
};

SubphraserIndex::SubphraserIndex()
    : root(MakeHolder<Node>()){
    prop_dont_sort = 0;
    prop_max_words_dist = 0;
}

SubphraserIndex::~SubphraserIndex() {
}

bool SubphraserIndex::SetProperty(const TStringBuf& prop_name, const TStringBuf& prop_value) {
    if(prop_name == "dont_sort") {
        prop_dont_sort = FromString<int>(prop_value);
    } else if(prop_name == "max_words_dist") {
        prop_max_words_dist = FromString<int>(prop_value);
    } else {
        return false;
    }

    return true;
}

const TStringBuf SubphraserIndex::GetPhraseData(const TStringBuf& phrase) const {
    const auto words = GetPhraseWords(phrase);

    if(words) {
        const Node* node = root.Get();

        for(const auto& word: words) {
            if (const auto it = node->edges.FindPtr(word)) {
                node = it;
            } else {
                return "";
            }
        }

        if (node->is_terminal) {
            return node->data;
        }
    }

    return "";
}

void SubphraserIndex::AddPhrase(const TStringBuf& phrase, const TStringBuf& data) {
    const auto words = GetPhraseWords(phrase);

    if(words) {
        TWriteGuard guard(Lock);
        TraceAddPhrase(words, 0, *root, data);
    }
}

void SubphraserIndex::GetSubphrases(const TStringBuf& phrase, TVector<TString>& result) const {
    const auto words = GetPhraseWords(phrase);

    if(words) {
        TReadGuard guard(Lock);
        TVector<TStringBuf> trace;
        trace.reserve(words.size());
        TraceGetSubphrases(words, 0, *root, trace, result);
    }
}

void SubphraserIndex::TraceAddPhrase(const TVector<TStringBuf>& words, unsigned pos, Node& node, const TStringBuf& data) {
    if(pos >= words.size()) {
        node.data = data;
        node.is_terminal = true;
    } else {
        const TString word(words[pos]);
        if(!node.edges.FindPtr(word)) {
            auto bufferNode = Node();
            TraceAddPhrase(words, pos + 1, bufferNode, data);
            node.edges.emplace(word, bufferNode);
        } else {
            TraceAddPhrase(words, pos + 1, node.edges[word], data);
        }
    }
}

void SubphraserIndex::TraceGetSubphrases(const TVector<TStringBuf>& words, unsigned pos, const Node& node, TVector<TStringBuf>& trace, TVector<TString>& result) const {
    if(node.is_terminal) {
        result.push_back(JoinSeq(TERMINAL_NODE, trace));
        result.push_back(node.data);
    }

    unsigned end_pos = (prop_max_words_dist && &node != root.Get()) ? std::min((unsigned)words.size(), pos + ((unsigned)prop_max_words_dist)) : words.size();
    for(unsigned i = pos; i < end_pos; i++) {
        if(const auto subNode = node.edges.FindPtr(words[i])) {
            trace.push_back(words[i]);
            TraceGetSubphrases(words, i + 1, *subNode, trace, result);
            trace.pop_back();
        }
    }
}

const TVector<TStringBuf> SubphraserIndex::GetPhraseWords(const TStringBuf& phrase) const {
    TVector<TStringBuf> words;
    Split(phrase, " ", words);
    if(!prop_dont_sort) {
        SortUnique(words);
    }
    return words;
}

void SubphraserIndex::SetPhraseOriginal(const TStringBuf& category, const TStringBuf& phrase, const TStringBuf& text) {
    original_map[TString(category)][TString(phrase)] = text;
}

const TStringBuf SubphraserIndex::GetPhraseOriginal(const TStringBuf& category, const TStringBuf& phrase) const {
    if (const auto catPtr = original_map.FindPtr(category)) {
        if (const auto phrasePtr = catPtr->FindPtr(phrase)) {
            return *phrasePtr;
        }
    }
    return "";
}
