#include "SubphraserServer.h"

#include <rt-research/broadmatching/scripts/cpp-source/common/Session.h>

#include <util/generic/string.h>
#include <util/string/split.h>
#include <util/string/join.h>

#include <library/cpp/logger/global/global.h>

#define DEBUG_MEMORY 0

SubphraserServer::SubphraserServer(int port)
    : Server(port)
{
}

void SubphraserServer::ExecCommand(const TStringBuf& command, Session* session) {
#if DEBUG_MEMORY
    struct rusage _usage;
    getrusage(RUSAGE_SELF, &_usage);
    DEBUG_LOG << "Incoming command, length=" << strlen(command) << ", maxrss= " << _usage.ru_maxrss;
#endif
    TVector<TStringBuf> cmd = StringSplitter(command).Split('\t');

    if (!cmd.size()) {
        session->SendResponse("ERROR: empty command");
    } else if (cmd[0] == "bye") {
        INFO_LOG << "Closing session";
        session->SendResponse("BYE");
        session->Stop();
    } else if (cmd[0] == "stop") {
        INFO_LOG << "Stopping server";
        session->SendResponse("BYE");
        session->Stop();
        Stop();
    } else {
        TStringBuilder responseBuilder;
        ExecIndexCommand(cmd, responseBuilder);

        if (!responseBuilder) {
            responseBuilder << "ERROR: unknown command '" << cmd[0] << "'";
        }
        session->SendResponse(responseBuilder.data());
    }
}

void SubphraserServer::ExecIndexCommand(const TVector<TStringBuf>& cmd, TStringBuilder& output) {
    if (cmd[0] == "add") {
        DEBUG_LOG << "Adding phrase `" << cmd[2] << "` to index `" << cmd[1] << "` category: `" << cmd[3] << "`";
        if (cmd.size() != 4) {
            output << "ERROR: 'add' takes 3 arguments";
        } else {
            indices[TString(cmd[1])].AddPhrase(cmd[2], cmd[3]);
            output << "OK";
        }
    } else if (cmd[0] == "propset") {
        DEBUG_LOG << "Setting property `" << cmd[2] << "` in index `" << cmd[1] << "` with value: `" << cmd[3] << "`";
        if (cmd.size() != 4) {
            output << "ERROR: 'propset' takes 3 arguments";
        } else {
            if (indices[TString(cmd[1])].SetProperty(cmd[2], cmd[3])) {
                output << "OK";
            } else {
                output << "ERROR: can't change property value";
            }
        }
    } else if (cmd[0] == "data") {
        DEBUG_LOG << "Getting phrase data `" << cmd[2] << "` from index `" << cmd[1] << "`";
        if (cmd.size() != 3) {
            output << "ERROR: 'data' takes 2 arguments";
        } else {
            output << "OK\t" << indices[TString(cmd[1])].GetPhraseData(cmd[2]);
        }
    } else if (cmd[0] == "get") {
        DEBUG_LOG << "Get subphrases for phrase `" << cmd[2] << "` in index `" << cmd[1] << "`";
        if (cmd.size() != 3) {
            output << "ERROR: 'get' takes 2 arguments";
        } else {
            TVector<TString> result;

            result.push_back("OK");
            if (const auto it = indices.FindPtr(cmd[1])) {
                it->GetSubphrases(cmd[2], result);
            }

            output << JoinSeq("\t", result);
        }
    } else if (cmd[0] == "getn") {
        DEBUG_LOG << "Get subphrases for phrases `" << cmd[2] << "` in index `" << cmd[1] << "`";
        if (cmd.size() != 3) {
            output << "ERROR: 'getn' takes 2 arguments";
        } else {
            char buff[30];
            TVector<std::string> result;

            result.push_back("OK");

            if (const auto it = indices.FindPtr(cmd[1])) {
                TVector<TString> phrases = StringSplitter(cmd[2]).Split(',');
                for (unsigned i = 0; i < phrases.size(); i++) {
                    TVector<TString> subphrases;
                    it->GetSubphrases(phrases[i], subphrases);
                    sprintf(buff, "%d", (int)subphrases.size());
                    result.push_back(buff);
                    for (unsigned j = 0; j < subphrases.size(); j++) {
                        result.push_back(subphrases[j]);
                    }
                }
            }

            output << JoinSeq("\t", result);
        }
    } else if (cmd[0] == "setorig") {
        DEBUG_LOG << "Setting dictionary form of phrase `" << cmd[3] << "` to index `" << cmd[1] << "` category: `" << cmd[2] << "` text: `" << cmd[4] << "`";
        if (cmd.size() != 5) {
            output << "ERROR: 'setorig' takes 4 arguments";
        } else {
            indices[TString(cmd[1])].SetPhraseOriginal(cmd[2], cmd[3], cmd[4]);
            output << "OK";
        }
    } else if (cmd[0] == "getorig") {
        DEBUG_LOG << "Getting dictionary form of phrase `" << cmd[3] << "` to index `" << cmd[1] << "` category: `" << cmd[2] << "`";
        if (cmd.size() != 4) {
            output << "ERROR: 'getorig' takes 4 arguments";
        } else {
            TVector<TStringBuf> result;

            result.push_back("OK");

            if (const auto it = indices.FindPtr(cmd[1])) {
                const auto text = it->GetPhraseOriginal(cmd[2], cmd[3]);
                if (text) {
                    result.push_back(text);
                }
            }

            output << JoinSeq("\t", result);
        }
    }
}
