#pragma once

#include <util/system/types.h>
#include <util/generic/maybe.h>
#include <util/generic/overloaded.h>

#include <variant>
#include <array>
#include <string>

struct TErrorMessage;

template<class Result, class Error>
class TResult : public std::variant<Result, Error> {
public:
    using std::variant<Result, Error>::variant; // inherit constructors
    TMaybe<Result> GetResult() const;
    TMaybe<Error> GetError() const;
};

namespace NPrivate {
    struct TStateStart {};
    struct TStateParsing {
        std::array<ui8, 4> parsedChars;
        uint8_t position;
        uint8_t length;
    };
}

class TUtf8Parser {
public:
    TUtf8Parser();

    // Contains 1 in `int` if symbol is parsed, second argument is the error message
    using TResult = TResult<int, TErrorMessage>;

    [[nodiscard]] TResult ProcessChar(ui8 symbol) noexcept;
    [[nodiscard]] TResult Finish() noexcept;

private:
    class TState : public std::variant<NPrivate::TStateStart, NPrivate::TStateParsing> {
    public:
        using TStart = NPrivate::TStateStart;
        using TParsing = NPrivate::TStateParsing;
        using std::variant<NPrivate::TStateStart, NPrivate::TStateParsing>::variant; // inherit constructors
    };
    
    TState state_; // inspired by Rust enums
};

template<class Result, class Error>
TMaybe<Result> TResult<Result, Error>::GetResult() const {
    return std::visit(TOverloaded {
        [](const Result& result) {
            return TMaybe<Result>(result);
        },
        [](const Error&) {
            return TMaybe<Result>();
        }
    }, *this);
}

template<class Result, class Error>
TMaybe<Error> TResult<Result, Error>::GetError() const {
    return std::visit(TOverloaded {
        [](const Result&) {
            return TMaybe<Error>();
        },
        [](const Error& error) {
            return TMaybe<Error>(error);
        }
    }, *this);
}
