#pragma once

#include "ampersand_parser.h"
#include "cycle_detector.h"
#include "utf8_parser.h"

#include <util/string/builder.h>

#include <clocale>
#include <fstream>
#include <stdexcept>
#include <string>
#include <unordered_set>
#include <vector>

#undef seed

struct TErrorMessage;

// Class for providing single error or single warning for perl
struct TErrorOrWarning {
    bool is_warning;
    TErrorMessage message;

    // Static constructors
    static TErrorOrWarning InternalError(const TStringBuf message, const TStringBuf message_ru);
    static TErrorOrWarning Error(const TStringBuf message, const TStringBuf message_ru);
    static TErrorOrWarning Warning(const TStringBuf message, const TStringBuf message_ru);
};

class XMLParser {
public:
    template <typename TSearchTagIterator>
    static TMaybe<TErrorOrWarning> Grep(
        const char* source_file_path,
        const char* target_file_path,
        TSearchTagIterator search_tag_first,
        TSearchTagIterator search_tag_last,
        bool strong_mode=false,
        const char* feed_type=""
    ) noexcept;

    template <typename TSearchIterator>
    static TMaybe<TErrorOrWarning> GrepGeneral(
        std::istream* source_stream,
        std::ostream* target_stream,
        TSearchIterator search_tag_first,
        TSearchIterator search_tag_last,
        bool is_strong_mode,
        const char* feed_type
    ) noexcept;

private:
    static char Normalize(char character);
    TErrorOrWarning ParserError(
        const TStringBuf message_ru,
        const TStringBuf message_en,
        bool add_line_column_number
    );
    TErrorOrWarning Utf8Error(const TErrorMessage& message);
    TErrorOrWarning ExpectedSymbolsError(const TStringBuf symbols_en, const TStringBuf symbols_ru);
    TErrorOrWarning ExpectedSymbolError(const char symbol);
    TErrorOrWarning AmpersandError(const TErrorMessage& message);
    TErrorOrWarning EofError();
    TErrorOrWarning WrongPathError();

    void ThrowOrMemoizeNewError(TErrorOrWarning error);

    XMLParser(std::istream* source_stream, std::ostream* target_stream, bool is_strong_mode, const char* feed_type);

    void GetChar();
    char current_char_;

    // DYNSMART-472 - предобработка блока <![CDATA[ ]]> и блока комментария
    enum class BufferState {in_comment, in_cdata, in_xml_code};
    void ReadBuffer();
    const unsigned long BUF_SIZE = 8192;
    bool is_EOF;
    bool is_strong_mode;
    bool is_search_for_warnings;
    unsigned long buffer_pos;
    BufferState buf_state;
    std::string end_buffer;
    std::string buffer;
    TMaybe<TErrorOrWarning> warning;

    TUtf8Parser utf8_parser;
    TAmpersandParser ampersand_parser;
    TCycleDetector cycle_detector;

    void GetNextChar();
    void GetSyntacticLexeme();
    void GetTagLexeme();
    void GetAttributeKeyLexeme();
    void GetAttributeValueLexeme(char opened_quot = '\"');
    std::string GetBodyLexeme();
    void ParseHeader();
    void ParseXML();
    void ParseAttributes();
    void ProcessOpenTagEvent();
    void ProcessCloseTagEvent();
    void PushSearchTag(const std::string& tag);
    void WriteFullTagsStack();
    const TMaybe<TErrorOrWarning>& GetWarning() const;

    std::istream* source_stream_;
    std::ostream* target_stream_;
    size_t line_index_;
    size_t column_index_;
    std::string current_lexeme_;
    std::vector<std::string> tags_stack_;
    std::vector<char> write_flags_stack_;
    std::unordered_set<std::string> search_tags_;
    std::unordered_set<std::string> offer_tags_;
    bool search_for_duplicates_;
    TString feed_type_;
};

template <typename TSearchIterator>
TMaybe<TErrorOrWarning> XMLParser::GrepGeneral(
    std::istream* source_stream,
    std::ostream* target_stream,
    TSearchIterator search_tag_first,
    TSearchIterator search_tag_last,
    bool is_strong_mode,
    const char* feed_type
) noexcept {
    try {
        XMLParser xml_parser(source_stream, target_stream, is_strong_mode, feed_type);
        for (auto search_tag_iterator = search_tag_first;
            search_tag_iterator < search_tag_last;
            ++search_tag_iterator) {
            xml_parser.PushSearchTag(*search_tag_iterator);
        }
        xml_parser.GetChar();

        // игнорируем символы перед первым <
        while (xml_parser.current_char_ != '<') {
            if (!xml_parser.is_EOF) {
                xml_parser.GetChar();
                continue;
            }
            return TErrorOrWarning::InternalError("text has no feed start symbols", "в тексте не найдены символы начала фида");
        }

        xml_parser.GetSyntacticLexeme();
        xml_parser.ParseHeader();
        xml_parser.ParseXML();

        if (const auto warning = xml_parser.GetWarning()) {
            return *warning;
        }
    } catch (TErrorOrWarning& e) {
        return e;
    } catch (std::exception& e) {
        return TErrorOrWarning::InternalError(e.what(), e.what());
    } catch (...) {
        return TErrorOrWarning::InternalError("unknown exception", "неизвестное исключение");
    }
    return Nothing();
}

template <typename TSearchIterator>
TMaybe<TErrorOrWarning> XMLParser::Grep(
    const char* source_file_path,
    const char* target_file_path,
    TSearchIterator search_tag_first,
    TSearchIterator search_tag_last,
    bool is_strong_mode,
    const char* feed_type
) noexcept {
    try {
        std::setlocale(LC_ALL, "ru_RU.UTF-8");
        std::setlocale(LC_NUMERIC, "en_US.UTF-8");
        std::ifstream source_stream(source_file_path);
        if (!source_stream.good()) {
            return TErrorOrWarning::Error(
                TStringBuilder{} << "can't read file `" << source_file_path << "`",
                TStringBuilder{} << "невозможно прочитать файл `" << source_file_path << "`"
            );
        }
        std::ofstream target_stream(target_file_path);
        if (!target_stream.good()) {
            return TErrorOrWarning::Error(
                TStringBuilder{} << "can't write file `" << target_file_path << "`",
                TStringBuilder{} << "невозможно записать в файл `" << target_file_path << "`"
            );
        }
        return XMLParser::GrepGeneral(&source_stream, &target_stream, search_tag_first, search_tag_last, is_strong_mode, feed_type);
    } catch (TErrorOrWarning& e) {
        return e;
    } catch (std::exception& e) {
        return TErrorOrWarning::InternalError(e.what(), e.what());
    } catch (...) {
        return TErrorOrWarning::InternalError("unknown exception", "неизвестное исключение");
    }
    return Nothing();
}
