use strict;
use warnings;
use utf8;
use open ':utf8';
no warnings 'utf8';
use Getopt::Long;


binmode(STDIN, ":utf8");
binmode(STDOUT, ":utf8");
binmode(STDERR, ":utf8");


use FindBin;
use lib "$FindBin::Bin/../../lib";
use lib "$FindBin::Bin/../../wlib";
use Utils::Hosts;
use Utils::Sys qw(save_json load_json);
use Data::Dumper;
use Project;
use JSON qw(to_json from_json);

my $proj = Project->new({});

main();

sub upload_url_to_s3 {
    my $task_id = shift;
    my $url = shift;
    my $login = shift;
    my $pass = shift;
    my $s3_bucket = shift;
    my $force_reload_feed = shift;
    # проверяем, что данный фид не лежит в s3, если лежит возвращаем ссылку на него
    if (!$force_reload_feed and $proj->read_sys_cmd("aws --endpoint-url=http://s3.mds.yandex.net s3 ls s3://$s3_bucket/${task_id}_feed", no_die => 1)) {
        return "https://$s3_bucket.s3.mds.yandex.net/${task_id}_feed"
    }

    my $feed_tmp_filepath = $proj->get_tempfile("${task_id}_feed", UNLINK => 1);
    my $page = $proj->page({ url => $url, name => "___NONE___" });
    if ($login) {
        $page->{login} = $login;
        $page->{pass} = $pass;
    }

    $page->timeout(3600);
    $page->{zora_big_files} = 1;
    $page->{zora} = 1;
    $page->result_file($feed_tmp_filepath);
    $page->tt; # записали в файл

    if (-e $feed_tmp_filepath and -s "$feed_tmp_filepath") {
        # для заливки нужно чтобы ключи лежали в правильном месте
        $proj->do_sys_cmd("aws --endpoint-url=http://s3.mds.yandex.net s3 cp $feed_tmp_filepath s3://$s3_bucket/${task_id}_feed");
        my $new_feed_url = "https://$s3_bucket.s3.mds.yandex.net/${task_id}_feed";
        $proj->log("$task_id: OK");
        unlink $feed_tmp_filepath;
        return $new_feed_url;
    } else {
        $proj->log("$task_id: Download failed; ignore this task");
        return;
    }
}

sub download_from_yt {
    my $source_table = shift;
    my $target_filepath = shift;
    $proj->yt_client->read_table_to_file($source_table, $target_filepath, 'json');
}

sub upload_to_yt {
    my $source_filepath = shift;
    my $target_yt_table = shift;
    $proj->yt_client->write_table_from_file($source_filepath, $target_yt_table, 'json');
}

sub task_id_second_field {
    my $task_type = shift;
    if ($task_type eq 'perf') {
        return 'GroupExportIDs';
    }
    elsif ($task_type eq 'dyn') {
        return 'BannerIDs';
    }
}

sub replace_feedurl_in_json {
    my $source_filepath = shift;
    my $target_filepath = shift;
    my $s3_bucket = shift;
    my $force_reload_feed = shift;
    my $task_type = shift;

    open(my $in_fh, "<", $source_filepath) or die('not open(');
    open(my $out_fh, ">", $target_filepath) or die('not open(');
    while (<$in_fh>) {
        my $row = from_json($_);
        my $task_id = join "_", $row->{OrderID}, @{from_json($row->{task_id_second_field($task_type)})};

        my $resource = from_json($row->{Resource});
        my $feed_url = $resource->{FeedUrl};
        my $login = $resource->{Login};
        my $password = $resource->{Password};


        my $new_feed_url = upload_url_to_s3($task_id, $feed_url, $login, $password, $s3_bucket, $force_reload_feed);
        if ($new_feed_url) {
            $resource->{FeedUrl} = $new_feed_url;
            delete $resource->{Login};
            delete $resource->{Password};

            $row->{Resource} = to_json($resource);
            my $res_row = to_json($row);
            print $out_fh "$res_row\n";
        }
    }
    close $in_fh;
    close $out_fh;
}

sub is_have_external_filters { # need for dyn
    die "not implemented";
}

sub parse_args {
    my %opt;

    GetOptions(
        \%opt,
        'input_yt|input-yt=s',
        'output_yt|output-yt=s',
        'task_type|task-type=s',
        's3_bucket|s3-bucket=s',
        'force_reload_feed|force-reload-feed',
        'help',
    ) or die "$!";
    my $help_message = join("\n\n",
        "get input yt tasks table, upload feeds in s3, upload result in yt output table",
        "for run need aws token https://wiki.yandex-team.ru/mds/s3-api/authorization/#upravlenieaccesskeys",
        "cat ~/.aws/credentials",
        "[default]",
        "aws_access_key_id = <access_key>",
        "aws_secret_access_key = <secret_key>",
        "",
        "\t--help: print this text",
        "\t--input-yt: input yt table with tasks",
        "\t--output-yt: output yt table with tasks (feedurl in s3)",
        "\t--task-type: task type [perf, dyn]",
        "\t--s3-bucket: s3 bucket for upload feed (default 'perf-feeds')",
        "\t--force-reload-feed: if not set then feed not update (not download and not upload in s3) if exist in s3 mds",
    );
    if ($opt{help}) {
        print "$help_message\n";
        exit 0;
    }
    die 'select --input-yt: input yt table with tasks' if !$opt{input_yt};
    die 'select --output-yt: input yt table with tasks' if !$opt{output_yt};
    die 'select --task-type: input tasks type' if !$opt{task_type};
    die 'select --s3-bucket: input s3 bucket for feeds' if !$opt{s3_bucket};
    return \%opt;
}

sub main {
    my $opts = parse_args();
    my $yt_in = $opts->{input_yt};
    my $yt_out = $opts->{output_yt};
    my $s3_bucket = $opts->{s3_bucket};
    my $force_reload_feed = $opts->{force_reload_feed};
    my $task_type = $opts->{task_type};

    my $taskjsons_file = $proj->get_tempfile('taskjsons', UNLINK => 1);
    my $taskjsons_file_out = $proj->get_tempfile('taskjsons_out', UNLINK => 1);
    download_from_yt($yt_in, $taskjsons_file);
    replace_feedurl_in_json($taskjsons_file, $taskjsons_file_out, $s3_bucket, $force_reload_feed, $task_type);
    upload_to_yt($taskjsons_file_out, $yt_out);
}
