#!/usr/bin/perl
use strict;
use warnings;

use open ':utf8';

use JSON qw(from_json);
use Getopt::Long;
use File::Path qw(make_path);
use File::Slurp qw(read_file);
use File::Copy qw(move);
use Parallel::ForkManager;

use FindBin;
use lib "$FindBin::Bin/../lib";

use Utils::Sys qw(
    save_json load_json
    dir_files
    handle_errors
    assert_curr_host
    dir_files
);
use Project;

# we do not require debug_env

handle_errors(DIE => {stack_trace => 1});

my %opt;
GetOptions(
    \%opt,
    'help|h',
    'task_type|task-type=s',
    'task_file|task-file|task_json|task-json=s@',
    'ext_tskv_gen|ext-tskv-gen=s',
    'task_dir|task-dir=s',
    'output_dir|output-dir=s',
    'relax',
);

if ($opt{help} or !$opt{task_type} or (!$opt{task_dir} and !@{$opt{task_file}})) {
    printf "Usage: $0 [Options] [files ...]\n";
    printf "Generate dyn/perf export_offers files\n";
    printf "Options:\n";
    printf "  --task-type           dyn|perf\n";
    printf "  --task-file           /path/to/task.json\n";
    printf "  --ext-tskv-gen        set ext_tskv_gen (for all tasks; as in DynGrpTask)\n";
    printf "  --output-dir          subj (default: cwd)\n";
    printf "  --task-dir           subj (default: None)\n";
    printf "                        dir with taskjson files\n";
    printf "                        ignore task-file arg if exist task-dir \n";
    printf "  --relax               do not die if some errors\n";
    exit;
}
my $type = $opt{task_type};
if ($type ne 'dyn' and $type ne 'perf') {
    die "Type must be 'dyn' or 'perf'\n";
}

$ENV{BANNERLAND_NO_IRON_GEN} //= '1';

my @task_files = ();

if ($opt{task_dir}) {
    printf "WARNING: task_dir selected; IGNORE  task-file arg\n" if @task_files;
    @task_files = map {"$opt{task_dir}/$_"} dir_files($opt{task_dir});
} else {
    @task_files = @{$opt{task_file}};
}

my $proj = Project->new({
    load_dicts => 1,
    load_minicategs_light => 1,
    use_comptrie_subphraser => 1,
    load_languages => [ qw(ru en tr) ],
    use_sandbox_categories_suppression_dict => 1,
    allow_lazy_dicts => 1,
});

# делается в CatalogiaMediaProject
$proj->dbh( $proj->catalogia_media_dbh );

my $failed = 0;
my $output_dir = $opt{output_dir} // '.';
my @res;
my $pm = Parallel::ForkManager->new(8);
$pm->run_on_finish(
    sub {
        my ($pid, $exit_code, $ident, $exit_signal, $core_dump, $data) = @_;
        if ($exit_code != 0) {
            $proj->log("process $pid failed!");
            $failed = "task $data->{task_id}";
        }
        if (defined $data) {
            my $meta = delete $data->{_meta};
            $proj->log("export-offers: finished task '$data->{task_id}' with exit code $exit_code, $meta->{elapsed} seconds elapsed, task_file '$meta->{task_file}', output in '$meta->{stderr}'");
        }
    }
);

for my $task_file (@task_files) {
    last if $failed and !$opt{relax};
    next if $opt{relax} and (!-f $task_file or -z $task_file);
    open my $task_fh, '<', $task_file
        or die "Can't open task file!";
    my $task_line = join(' ', map { chomp; $_ } <$task_fh>);
    close $task_fh;
    my $task_inf = from_json($task_line);

    $pm->start() and next;
    
    my $task_obj;
    $task_obj = $proj->get_task_obj($type, $task_inf);
    make_path($task_obj->dir) if !-d $task_obj->dir;

    my $new_outfile = $task_obj->dir . "/output";
    $proj->log("export-offers: started task for task_file '$task_file', output redirected to: '$new_outfile.log', '$new_outfile.err'");

    close(STDOUT);
    close(STDERR);
    
    open(STDOUT, "> $new_outfile.log");
    open(STDERR, "> $new_outfile.err");
    
    my $started = time();
    my $task_id = $task_obj->task_id;
    $task_obj->{_begin_time} = time;

    my $eo_file = "$output_dir/$task_id.export_offers.tskv";

    if ($type eq 'dyn') {
        # cache some data for usage in YT
        $task_obj->redirect_domain;
        $task_obj->specfilters;
    }

    if ($opt{ext_tskv_gen}) {
        $proj->log("set ext_tskv_gen: $opt{ext_tskv_gen}");
        $task_obj->{ext_tskv_gen} = $opt{ext_tskv_gen};
    }

    my $res = $task_obj->worker_generate(
        export_offers_par => {
            local_output => $eo_file,
            timeout => 3600 * 24,
        },
    );

    if ($res->{export_info}) {
        $proj->log("added file: $eo_file");
    } else {
        my $msg = "Can't export data for $task_id!";
        $msg .= " error: $res->{error}" if $res->{error};
        if (!$opt{relax}) {
            die "$msg\n";
        } else {
            $proj->log("RELAX (no_die): $msg");
        }
    }

    my $elapsed = time() - $started;
    
    $proj->log("task $task_file done, $elapsed seconds elapsed");

    $pm->finish(0, {
        task_id => $task_id,
        task_inf => $task_obj->taskinf,
        _meta => {
            task_file => $task_file,
            stderr => "$new_outfile.err",
            elapsed => $elapsed,
        },
    });
}

$pm->wait_all_children();
if ($failed) {
    my $msg = "Some of tasks failed: $failed!";
    if (!$opt{relax}) {
        die "$msg\n";
    } else {
        $proj->log("RELAX (no_die): $msg");
    }
}
$proj->log("all tasks done");
