#!/usr/bin/perl
use strict;
use warnings;

use utf8;
use open ':utf8';

use File::Basename qw(basename);
use Getopt::Long;

use FindBin;
use lib "$FindBin::Bin/../lib";
use lib "$FindBin::Bin/../wlib";
use lib "$FindBin::Bin/../cpan";
use CatalogiaMediaProject;

use Utils::Sys qw(
    handle_errors
    load_json save_json
    print_log
);

handle_errors(DIE => {stack_trace=>1});

my %opt;
GetOptions(
    \%opt,
    'task_type|task-type=s',
    'task_id|task-id=s@',
    'task_table|task-table=s',
    'output_file|output-file=s@',
    'output_dir|output-dir=s',
    'relax',
    'help|h',
);
if ($opt{help}) {
    printf "Usage: $0 [Options]\n";
    printf "Get taskjson files\n";
    printf "Options:\n";
    printf "    --task-type     dyn|perf\n";
    printf "    --task-id       subj\n";
    printf "    --task-table    source YT table for tasks\n";
    printf "    --output-file   subj (it only work if --task-id selected)\n";
    printf "    --output-dir    subj\n";
    printf "    --relax         do not die if some tasks not found\n";
    exit;
}

my $task_type = $opt{task_type}
    or die "task-type not defined!";

my $proj = Project->new({ memlog => 1, });
my $task_out_dir = $opt{output_dir};
$proj->do_sys_cmd("mkdir -p $task_out_dir") if ($task_out_dir);

sub get_output_filename {
    my $task_id = shift;
    return "$task_out_dir/${task_id}_taskjson" if ($task_out_dir);;
    return shift @{$opt{output_file}};
}

my @not_found;
my @created;
my $tsk = $proj->get_task_obj($task_type, {});

my $tasks_table = $opt{task_table} // $tsk->get_default_input_tasks_yt_table;
if ($opt{task_id}) {
    my @keys;
    for my $task_id (@{$opt{task_id}}) {
        my ($OrderID, $GroupingID) = split /_/, $task_id;
        push @keys, "(${OrderID}u, ${GroupingID}u)";
    }
    if (@keys > 1) {
        $ENV{YT_CONFIG_PATCHES} = '{read_retries={enable=%false;allow_multiple_ranges=%true}};';
    }
    $tasks_table .= '[' . join(',', @keys) . ']';
}

my $task_id_val = {};
my $tasks = $tsk->get_all_tasks_from_yt_table($tasks_table);
for my $task (@$tasks) {
    my $obj = $proj->get_task_obj($task_type, $task);
    my $task_id = $obj->task_id;
    $task_id_val->{$task_id} = $task;
}

my @needed_task_ids;
if ($opt{task_id}) {
    @needed_task_ids = grep {$task_id_val->{$_}} @{$opt{task_id}};
    @not_found = grep {!$task_id_val->{$_}} @{$opt{task_id}};

    if (@not_found) {
        my $msg = "Tasks not found for ids: " . join(', ', @not_found);
        if (!$opt{relax}) {
            die "$msg\n";
        } else {
            print_log("RELAX (no_die): $msg");
        }
    }
}
else {
    @needed_task_ids = keys $task_id_val;
}

for my $task_id (@needed_task_ids) {
    my $output_file = get_output_filename($task_id);
    my $task = $task_id_val->{$task_id};
    save_json($task, $output_file);
    push @created, $output_file;
}

if (@not_found) {
    my $msg = "Tasks not found for ids: @not_found";
    if (!$opt{relax}) {
        die "$msg\n";
    } else {
        print_log("RELAX (no_die): $msg");
    }
}
for my $file (@created) {
    print_log("Created file: $file");
}
print_log("get tasks done!");
