#!/usr/bin/perl -w
use strict;
use utf8;

use JSON qw(from_json to_json);
use Getopt::Long;
use File::Slurp qw(write_file);
use Time::HiRes qw(gettimeofday);

use FindBin;
use lib "$FindBin::Bin/../lib";
use lib "$FindBin::Bin/../wlib";

use Utils::Sys qw{handle_errors print_log dir_files load_json};
use Project;

handle_errors();

my %opt;
GetOptions(\%opt,
    'help',
    'task_file|task-file=s@',
    'task_dir|task-dir=s',
    'output_dir|output-dir=s',
);
if ($opt{help}) {
    printf "Usage: $0 [Options]\n";
    printf "Generate offer previews for perf tasks\n";
    printf "    --task-file     file with task json (multiple option)\n";
    printf "    --task-dir     directory with task json files (adds to task-file)\n";
    printf "    --output-dir    subj (if not specified, write to stdout)\n";
    exit (0);
}

my $proj = Project->new({
    load_dicts => 1,
    load_minicategs_light => 1,
    use_comptrie_subphraser => 1,
    load_languages => [ qw(ru en tr) ],
    use_sandbox_categories_suppression_dict => 1,
    allow_lazy_dicts => 1,
});

my @task_files = @{$opt{task_file}};
push @task_files, map {"$opt{task_dir}/$_"} dir_files($opt{task_dir}) if $opt{task_dir};
if (!@task_files) {
    die "Can't get input tasks!";
}

# https://a.yandex-team.ru/arc/trunk/arcadia/direct/perl/protected/ppcFeedToBannerLand.pl
# https://a.yandex-team.ru/arc/trunk/arcadia/rt-research/broadmatching/scripts/broadmatching-server/fcgi-http-server
for my $task_file (@task_files) {
    my $task_inf = load_json($task_file);
    my $task_obj = $proj->perftask($task_inf);
    my $task_id = $task_obj->task_id;
    my $start = gettimeofday();

    my $task_feedparams = $task_obj->get_feedparams;
    delete $task_feedparams->{filters};

    my %other_feedparams = (
        download_timeout => 1200,  # as in fcgi-http-server
        max_file_size => 512 * 1024 * 1024,  # 512Mb
        max_file_size_type => 'bytes',
    );
    my $fd = $proj->feed({%$task_feedparams, %other_feedparams});

    my %inf_params = (gen_previews => 1);
    my $res = $fd->yml2directinf(\%inf_params);
    my $time = gettimeofday() - $start;
    
    if ($opt{output_dir}) {
        my $output_file = "$opt{output_dir}/$task_id.json";
        write_file($output_file, {binmode => 'utf8'}, $res);
        print_log("created file: $output_file");
    } else {
        print_log("got previews for task $task_id:");
        print_log("elapsed time: $time sec");
        print STDOUT $res, "\n";
    }
}
