#!/usr/bin/perl
use utf8;
use strict;
use warnings;
no warnings "utf8";

use open ':utf8';

use JSON;
use Getopt::Long;

use FindBin;
use lib "$FindBin::Bin/../lib";

use Project;
use Utils::Sys qw(
    handle_errors print_log
);
use Utils::Funcs qw(decode_tsv);


my %kwargs;
GetOptions(
    \%kwargs,
    'task_type|task-type=s',
    'help',
    'eo_file|eo-file=s',
    'tao_file|tao-file=s',
    'tao_table|tao-table=s',
    'save_tao|save-tao',
);


if ($kwargs{help} or (!$kwargs{eo_file} and !$kwargs{to_file} and !$kwargs{tao_table})) {
    print "Usage: $0 [Options]\n";
    print "Run native generation using export-offers-file\n",
    print "Options:\n";
    print "  --task-type    dyn or perf (default: perf)\n";
    print "  --eo-file      export_offers file\n";
    print "  --tao-file     tasks_and_offers_file (joined export_offers, in dsv format)\n";
    print "  --tao-table    path to tasks_and_offers table on YT\n";
    print "  --save-tao     do not remove tao file\n";
}

handle_errors(DIE => {stack_trace => 1});

my $task_type = $kwargs{task_type} // 'perf';

my $tao_file;
if ($kwargs{tao_file}) {
    $tao_file = $kwargs{tao_file};
} else {
    $tao_file = Utils::Sys::get_tempfile('tao_file', DIR => '.', UNLINK => !$kwargs{save_tao});
    if ($kwargs{eo_file}) {
        join_tasks_and_offers($kwargs{eo_file}, $tao_file);
    } elsif ($kwargs{tao_table}) {
        dowload_export_offers($kwargs{tao_table}, $tao_file);
    }
}

print_log("will use tasks_and_offers file: $tao_file");
run_native_gen($task_type, $tao_file);
exit(0);


sub is_task_line {
    my $line = shift;
    return 1 if grep { /^task_inf=/ } split /\t/, $line;
    return 0;
}

sub join_tasks_and_offers {
    my $eo_file = shift;
    my $tao_file = shift;

    my $task_line;
    open my $ifh, '<', $eo_file
        or die "Can't open eo file `$eo_file': $!";
    while (<$ifh>) {
        chomp;
        my $line = $_;
        if (is_task_line($line)) {
            $task_line = $line;
            last;
        }
    }
    if (!defined $task_line) {
        die "Can't get task line!";
    }

    seek($ifh, 0, 0);
    open my $ofh, '>', $tao_file
        or die "Can't write to tao file `$tao_file': $!";
    while (<$ifh>) {
        chomp;
        my $line = $_;
        next if is_task_line($line);
        print $ofh $task_line, "\t", $line, "\n";
    }
    close $ifh;
    close $ofh;
}

sub dowload_export_offers {
    my $yt_path = shift;
    my $file_path = shift;
    my $proj = Project->new();
    $proj->yt_client->read_table_to_file($yt_path, $file_path, 'dsv');
}

sub run_native_gen {
    my $task_type = shift;
    my $file_export_offers = shift;

    my $proj = Project->new({
        load_dicts => 1,
        load_minicategs_light => 1,
        use_comptrie_subphraser => 1,
        load_languages => [ qw(ru en tr) ],
        use_sandbox_categories_suppression_dict => 1,
        allow_lazy_dicts => 1,
    });
    my $json_obj = JSON::XS->new->utf8(0);
    my $r;
    open my $fh, "<", $file_export_offers or die("Error! Can't open $file_export_offers, reason is $!");

    while (my $line = <$fh>) {
        chomp $line;
        $r = {};
        for my $kv (split /\t/, $line) {
            my ($k, $v) = split /=/, $kv, 2;
            $r->{$k} = decode_tsv($v);
        }
        my $task_inf = $json_obj->decode($r->{task_inf});
        my $product_inf = $json_obj->decode($r->{product_inf});
        my $pt_class = $r->{product_class};
        my $ppar = $json_obj->decode($r->{ppar});

        my $task = $proj->get_task_obj($task_type, $task_inf);

        my $pt = $proj->product_by_classname($pt_class, { proj => $proj, %$product_inf }, no_init => 1);
        $pt->class_init(proj => $proj);

        $ppar->{ctx}->{timer} = $proj->get_new_timer() if ($task_type ne 'dyn');

        $ppar->{'ext_data'} = [];  # записываем пустой массив в ext_data, чтобы не генерировать external
        my $result = $task->process_offer($pt, %$ppar);
        $proj->log("generated native:");
        $proj->log_dump($result);
        $proj->log("===================== END =====================");
    }
}
