#!/usr/bin/perl

use strict;
use warnings;
use utf8;
 
use open ":utf8";
no warnings "utf8";

binmode (STDIN, ":utf8");
binmode (STDOUT, ":utf8");
binmode (STDERR, ":utf8");

use Getopt::Long;
use File::Slurp qw(read_file);
use Encode qw(decode_utf8);

use FindBin;
use lib "$FindBin::Bin/../lib";
use lib "$FindBin::Bin/../wlib";

use Utils::Common;
use CatalogiaMediaProject;
use Utils::Sys qw(handle_errors assert_curr_host);

handle_errors(DIE => {stack_trace => 1});

my $USAGE = join(
    "\n",
    "",
    "Usage: `BM_USE_PATCH=debug_env $0 --task-type perf --task-json file.json`",
    "",
    "Options:",
    "    --task-type perf/dyn/dyngrp - task type (mandatory)",
    "    --task-json /path/to/json.file - path to taskjson (mandatory), examples of data: https://paste.yandex-team.ru/426888",
    "    --use-task-logs - write task logs to separate files, like in production",
    "    --no-cleanup - do not remove task info from MySQL before task run",
    "    --ignore-external-sources - add 'ignore_external_sources' option to taskinf",
    "    --use-comptrie-subphraser - add 'use_comptrie_subphraser' option to \$proj",
    "",
    "Consider doing `rm -rf temp/dyn_banners` or `rm -rf temp/perf_banners` every time before you run this script: that will clean any file cache that could be there",
    "",
    "",
);



my %kvargs;
GetOptions(
    \%kvargs,
    'task_type|task-type=s',
    'task_json|task-json=s',
    'use_task_logs|use-task-logs',
    'ignore_external_sources|ignore-external-sources',
    'no_cleanup|no-cleanup',
    'use_comptrie_subphraser|use-comptrie-subphraser',
) or die join("\n", "Error in command line arguments", $USAGE);


unless ($kvargs{task_json} && $kvargs{task_type}) {
    die join("\n", "Provide all mandatory options", $USAGE);
}

unless (assert_curr_host(allow => "nothing")) {
    die join("\n", "You do not want to run this script without debug_env patch", $USAGE);
}

if (1) {
    require Cwd;
    my $bl_kit_path = Cwd::abs_path("$FindBin::Bin/../../../bannerland/tools/bl-kit.py");
    printf STDERR "________________________________________________________________________________________________\n";
    printf STDERR "| Script run_task.pl is deprecated, please use bl-kit:                                |\n";
    printf STDERR "|                                                                                              |\n";
    printf STDERR "| ===> %80s   <=== |\n", $bl_kit_path;
    printf STDERR "[______________________________________________________________________________________________]\n";
    printf STDERR "\n";
}

run_task(
    _get_proj(%kvargs),
    %kvargs,
);

exit(0);


sub _get_proj {
    my %opts = @_;
    
    return CatalogiaMediaProject->new({
        indcmd => 1, 
        load_dicts => 1,
        load_minicategs_light => 1,
        use_comptrie_subphraser => $opts{use_comptrie_subphraser},
        no_auth => 1,
        no_form => 1,
        nrmsrv => 0, 
    })
}

sub run_task {
    my $proj = shift;
    my %opts = @_;

    my $task_type = $opts{task_type} // die "provide task_type";
    my $task_json_fname = $opts{task_json} // die "provide task_fname";
    my $use_task_logs = $opts{use_task_logs} // 0;
    my $ignore_external_sources = $opts{ignore_external_sources};
    my $no_cleanup = $opts{no_cleanup};

    my $task_data = get_task_data($proj, read_file($task_json_fname));
    $task_data->{ignore_external_sources} = $ignore_external_sources if defined $ignore_external_sources;

    my $task = get_task_obj(
        $proj,
        task_type => $task_type,
        task_data => $task_data,
    );

    task_db_cleanup($task) unless $no_cleanup;

    $task->{use_task_logs} = $use_task_logs;
    $task->make_task();

    return 1;
}

sub get_task_data {
    my $proj = shift;
    my $task_json = shift;

    my $task_data = $proj->json_obj->decode(decode_utf8($task_json));

    # если в json лежал массив из одного элемента, то это ОК
    if (ref($task_data) eq 'ARRAY') {
        if (scalar(@$task_data) == 1) {
            $task_data = $task_data->[0];
        } else {
            die "task_json must contain exactly one task";
        }
    }

    if (ref($task_data) ne 'HASH') {
        die "bad task_data";
    }

    return $task_data;
}

sub task_db_cleanup {
    my $task = shift // die "provide task";
    my $proj = $task->proj();
    
    if ($task->isa("BM::BannersMaker::Tasks::DynGrpTask")) {
        for my $dyntask_json (@{ $task->tasks() }) {
            task_db_cleanup($proj->dyntask($dyntask_json));
        }
    } else {
        my $task_table = $task->tablename();
        my $task_id = $task->task_id();

        $proj->log("Will delete task '$task_id' from table '$task_table'");
        $proj->bannerland_dbh->do("delete from $task_table where TaskID = '$task_id'");
    }

    return 1;
}

sub get_task_obj {
    my $proj = shift;
    my %opts = @_;

    my $task_type = $opts{task_type} // die "provide task_type";
    my $task_data = $opts{task_data} // die "provide task_data";
    
    if ($task_type eq 'perf') {
        die 'run_task.pl disabled for perf, use bl-kit';
    } elsif ($task_type eq 'dyn') {
        return $proj->dyntask($task_data);
    } elsif ($task_type eq 'dyngrp') {
        return $proj->dyngrptask($task_data);        
    } else {
        die "unknown task_type '$task_type'";
    }
}
