#!/usr/bin/perl -w
use strict;
use utf8;

# получение нужных ресурсов по rsync

use Getopt::Long;
use Sys::Hostname;

use FindBin;
use DBI;
use lib "$FindBin::Bin/lib";
use Utils::Sys qw(
    get_file_lock
    wait_for_file_lock
    release_file_lock
    print_log
    print_err
    handle_errors
);
use Project;
use BM::Resources;
use Utils::Hosts qw(get_host_role);
use Utils::Common;
use BM::SolomonClient;

handle_errors();

my @errors;

my %opt;
GetOptions(\%opt, 'list=s', 'tbl_list=s', 'producer_host=s', 'help|h', 'waitlock', 'ipv4');
if ($opt{help}) {
    printf "Usage: get-resources.pl [Options]\n";
    printf "Options:\n";
    printf "  Resources options:\n";
    printf "    --list=R1,R2,...      comma-separated list of resources\n";
    printf "      OR\n";
    printf "    --tbl_list=T1,T2,...  comma-separated list of db resources\n";
    printf "    \n";
    printf "    --producer_host=HOST  producer host\n";
    printf "  Other options:\n";
    printf "    --waitlock            Wait while the file lock is free\n";
    printf "    --ipv4                Force IPv4\n";
    printf "    --help\n",
    exit;
}

if ($opt{waitlock}) {
    print_err("wait_for_file_lock ...");
    wait_for_file_lock()
        or die "Could not wait_for_file_lock!";
} else {
    get_file_lock()
        or exit(0);
}

print_err('get-resources ...');

my $proj = Project->new();

# простые ресурсы
my %res_par;

$res_par{rsync_add_opt} = ['-W'];  # убираем delta-алгоритм

my (@res_list, @res_tbl_list);
if ($opt{list}) {
    @res_list = split /,/, $opt{list};
    $res_par{bwlimit} = 1e6;
} elsif ($opt{tbl_list}) {
    @res_tbl_list = split /,/, $opt{tbl_list};
    $res_par{bwlimit} = 1e6;
} else {
    # production mode
    @res_list = get_required_resources($proj);
    @res_tbl_list = get_required_resources_db();
}

if ($opt{producer_host}) {
    $res_par{host} = $opt{producer_host};
}

if ($opt{ipv4}) {
    push @{$res_par{rsync_add_opt}}, '-4';
}

for my $resource (@res_list) {
    print_err("resource: $resource ...");
    my $result = get_resource($resource, %res_par);
    print_err("resource: $resource result: $result");
    push @errors, $resource   if !$result;
}

# mysql tgz ресурсы
my %opt_upload;
# TODO перенести настройки в Hosts.pm
if (get_host_role() eq 'bmfront' or get_host_role() eq 'catalogia-media-gen-test') {
    # Не храним Done-таблицы
    $opt_upload{cleanup} = 1;
}

for my $resource_name (@res_tbl_list) {
    print_err("tbl_resource: $resource_name ...");
    my $result = get_resource($resource_name, %res_par);
    print_err("tbl_resource: $resource_name result:$result");
    push @errors, $resource_name   if !$result;
}

if (@errors) {
    print_err("WARN: failed resources: " . join(" ", @errors) );
} else {
    print_err('GET_RESOURCES_OK');
    BM::SolomonClient->new()->set_success_script_finish("get-resources");
}
print_err('get-resources done!');
release_file_lock();
exit(@errors ? 1 : 0);
