package BM::BMClient::BenderClient;

use strict;

use utf8;
use open ':utf8';
no warnings 'utf8';
use base qw(ObjLib::ProjPart);

use IO::Select;
use Data::Dumper;
use File::Copy qw/ /;
use List::Util qw(min);
use Time::HiRes qw/gettimeofday tv_interval/;
use BM::Phrase;
use Utils::Common;
use Utils::Sys qw(switch_to_ipv4 switch_to_ipv6);

########################################################
# Интерфейс
########################################################

########################################################
# Инициализация
########################################################

__PACKAGE__->mk_accessors(qw(
    port
    host
    temp_dir
    work_dir
    server_dir
    data_file
    encoded_file
    index_file
    binary_file
));

sub init {
    my ($self) = @_;

    $self->{socket} = 0;
}

sub is_connected {
    my ($self) = @_;

    return $self->{socket};
}

sub connect_once {
    my ($self) = @_;

    return if $self->is_connected;

    switch_to_ipv4() if $self->{is_ipv4};
    my $socket = IO::Socket::INET->new(
        PeerAddr => $self->host, 
        PeerPort => $self->port, 
        Proto => "tcp",
        Timeout => 5
    );
    switch_to_ipv6() if $self->{is_ipv4};

    if($socket) {
        if(!$socket->connected) {
            $socket = 0;
            return;
        }

        binmode($socket, ":utf8");

        $self->log("bender: connected to ".$self->host.":".$self->port);

        $self->{socket} = $socket;
    }
}

sub connect {
    my ($self) = @_;
    my $socket;
    my $num_attempts = 0;
    my $max_attempts = $self->{max_attempts} || 1000;
    my $reconnect_sleep = $self->{reconnect_sleep} || 10;
    
    while(!$self->is_connected && $num_attempts < $max_attempts) {
        if($num_attempts) {
            $self->log("bender: can't connect ($num_attempts attempts)");
            sleep($reconnect_sleep);
        }
        $self->connect_once;
        $num_attempts++;
    }

    if(!$self->is_connected) {
        $self->log("can't connect to bender at " . $self->host);
        return 0;
    }

    return 1;
}

sub disconnect {
    my ($self) = @_;

    close($self->{socket}) if $self->{socket};
    $self->{socket} = 0;

    $self->log("bender: disconnected");
}

sub exec_command {
    my ($self, $cmd) = @_;
    my $resp = undef;
    my $max_exec_attempts = $self->{max_exec_attempts} || 3;

    while(!defined($resp) && $max_exec_attempts--) {
        $resp = eval { $self->exec_command_once($cmd) };
        if ($@) {
            warn "exec_command_once($cmd) failed: $@";
            sleep 20;
        }
    }
    return $resp;
}

sub exec_command_once {
    my ($self, $cmd) = @_;
    my $num_bytes;

    if(!$self->connect) {
        return undef;
    }
    
    $num_bytes = $self->{socket}->send("$cmd\n");
    if(!defined($num_bytes)) {
        $self->disconnect;
        $self->log("Socket::send() failed");
        return undef;
    }

    my @resp_chunks = ();
    while(!@resp_chunks || $resp_chunks[-1] !~ /\n/) {
        my $resp;

        my @ready = IO::Select->new($self->{socket})->can_read(10);
        if(!@ready) {
            $self->log("can_read timeout");
            $self->disconnect;
            return undef;
        }

        if(!defined($self->{socket}->recv($resp, 1024)) || !$resp) {
            $self->disconnect;
            $self->log("Socket::recv() failed");
            return undef;
        }

        push @resp_chunks, $resp;
    }

    my $full_resp = join "", @resp_chunks;
    chomp $full_resp;
    $self->disconnect if $full_resp eq "BYE";
    if($full_resp =~ /^ERROR/) {
#        $self->log($full_resp);
        $self->disconnect;
        return undef;
    }

    return $full_resp;
}

1;

