package BM::BMClient::BenderPattern;

use strict;

use utf8;
use open ':utf8';
no warnings 'utf8';
use base qw(ObjLib::ProjPart);
use Data::Dumper;
use Utils::Array qw (array_equal);

# Расширенная функциональность Pattern.cpp
# ручка бендера может искать только по токенам, и не знает порядок слов в баннере
# Также в индексе бендера есть только мультиворды длины 2. Если нужно производить поиск мультивордов длины 3 и более,
# то мультиворды разбиваются на биворды, происходит поиск по индексу и проверяется на соответствие
# паттерну в этом модуле
#
# Например при поиске по <доставка пицца 24> происходит разбиение на токены __mw_доставка_пицца __mw_пицца_24
# Но по таким токенам некоторые баннеры будут несоответствовать начальному паттерну,
# например "пицца 24 часа. доставка пиццы бесплатно"

sub init {
    my ($self) = @_;

    my $proj = $self->proj;
    my $text = $self->{text};

    # Нормализуем слова не меняя их порядка и не удаляя остальных символов (' ', '/', '[', ']', '<', '>')
    my $re_symbols = '[\[\]<>\/]';
    $text =~ s/($re_symbols)/ $1 /g; # Добавляем пробелы вокруг токенов, чтоб можно было сплитить
    $text = join ' ', map { $_ =~ /$re_symbols/ ? $_ : $proj->phrase($_)->norm_phr_with_stops} split ' ', $text;
    $self->{norm_text} = $text;

    $text =~ s/<\s+/</g;  # нужно для следующего шага, чтобы внутри угловых скобок пробелы были только между словами
    $text =~ s/\s+>/>/g;
    # Заменяем <w1 w1 w3> на __mw_w1_w2_w3
    $text =~ s/ (?=[^<>]+>)/_/g;
    $text =~ s/<([^<>]+)>/__mw_$1/g;

    my $groups = [];
    my @server_groups = ();

    my @atoms = $text =~ /\[([^\]]+)\]/g;
    $text =~ s/(\[[^\]]+\])/ /g;

    # все слова вне скобок считаем как одну группу
    my $is_atom = 0;
    for my $group_text ($text, @atoms) {
        my @group = ();
        my @server_parts = ();
        my $has_empty_part = 0;
        for my $part_text (split '\/', $group_text) {
            my $norm_part_text = $proj->phrase($part_text)->norm_phr_ordered; # удаляем стоп-слова
            my @part = ();
            my @server_words = ();
            for my $word (split ' ', $norm_part_text) {
                if ($word =~ /^__mw_/) {
                    $word = substr $word, 5;
                    my @mw_words = split '_', $word;
                    if (scalar(@mw_words) >= 2) {
                        push @part, \@mw_words;
                        for my $i (0..$#mw_words-1) {
                            push @server_words, "__mw_$mw_words[$i]_$mw_words[$i+1]";
                        }
                    } else {
                        push @part, $word;
                        push @server_words, $word;
                    }
                } else {
                    push @part, $word;
                    push @server_words, $word;
                }
            }
            if (@server_words) {
                push @server_parts, join ' ', @server_words;
            } else {
                $has_empty_part = 1;
            }
            push @group, \@part;
        }
        push @$groups, \@group;
        my $server_group_text = join '|', @server_parts;
        $server_group_text = "($server_group_text)" if $is_atom;
        $server_group_text = "$server_group_text?" if $has_empty_part && $is_atom;
        push @server_groups, $server_group_text;
        $is_atom = 1;
    }

    # Структура groups:
    # группа - это текст в квадратных скобках или весь текст вне скобок
    # например, "пицца [<24 часа>/круглосуточно] москва" - это две группы "пицца москва" и "<24 часа>/круглосуточно"
    # каждая группа делится на части разделителем / (может быть только одна часть)
    # каждая часть состоит из слов, каждое слово это строка или массив строк (в случае мультислова)
    # То есть "пицца [<24 часа>/круглосуточно] москва" парсится в [[[пицца, москва]],[[[24, час]],[круглосуточно]]]
    $self->{groups} = $groups;

    # server_pattern это паттерн пробразованный в формат Pattern.cpp
    # квадратные скобки заменяются на круглые, / на |, мультиворды разбиваются на биворды,
    # если есть пустая часть то после скобок добавляется ?
    $self->{server_pattern} = join ' ', @server_groups;
    $self->{server_pattern} =~ s/^\s+//;
    $self->{server_pattern} =~ s/\s+$//;
}

sub is_match() {
    my ($self, $texts) = @_;
    my $norm_texts = [ map {[ $self->proj->phrase($_)->normalize(norm => 1) ]} @$texts ];
    my $dict_words = { map {$_ => 1} map {@$_} @$norm_texts };

    GROUP:
    for my $group (@{$self->{groups}}) {
        for my $part (@$group) {
            my $all_words = 1;
            WORDS:
            for my $word (@$part) {
                if (ref($word) eq 'ARRAY') {
                    my @mw_words = @$word;
                    my $ln_mw = scalar(@mw_words);
                    for my $words (@$norm_texts) {
                        my $ln = scalar(@$words);
                        next if $ln < $ln_mw;
                        for my $i (0..$ln-$ln_mw) {
                            next WORDS if array_equal(\@mw_words, [@$words[$i..$i+$ln_mw-1]]);
                        }
                    }
                    $all_words = 0;
                    last;
                } else {
                    unless ($dict_words->{$word}) {
                        $all_words = 0;
                        last;
                    }
                }
            }
            next GROUP if $all_words;
        }
        return 0;
    }

    return 1;
}

1;