package BM::BMClient::CdictClient;

use strict;

use utf8;
use open ':utf8';
no warnings 'utf8';

use base qw(ObjLib::ProjPart);

use IO::Select;
use Socket;
use IO::Socket::INET6;
use List::Util qw(min);

use Utils::Sys qw(get_tempfile);

########################################################
# Интерфейс
########################################################

########################################################
# Инициализация
########################################################

__PACKAGE__->mk_accessors(qw(
    port
    host
    server_dir
    temp_dir
    work_dir
    dict_file
    index_file
    single_file
    raw_files_dir
    norm_config_file
    info_file
));

my $MIN_TIMEOUT = 15;
my $MAX_TIMEOUT = 60;
my $TIMEOUT_DELTA = 15;

my %cacheable_cmds = (
    'get'      => 1,
    'getnorm'  => 1,
    'getnormq' => 1,
    'getsnorm' => 1,
);

sub init {
    my ($self) = @_;

    $self->{socket} = 0;
    $self->{max_connect_attempts} = 500;
    $self->{wide_phrase_flag} = 1;
    $self->{bad_phrase_flag} = 2;
    $self->{getnorm_cmd} = "getnorm";
}

sub is_connected {
    my ($self) = @_;

    return $self->{socket};
}

sub connect_once {
    my ($self) = @_;

    return if $self->is_connected;

    my $socket = IO::Socket::INET->new(
        PeerAddr => $self->host,#inet_ntoa(inet_aton($self->host)),
        PeerPort => $self->port,
        Proto => "tcp",
        Timeout => 5
    );

    if(!$socket || !$socket->connected) {
        $socket = 0;

        $self->log("cdict: can't connect to ".$self->host.":".$self->port." ($!)");
    } else {
        binmode($socket, ":utf8");

        $self->log("cdict: connected to ".$self->host.":".$self->port);

        $self->{socket} = $socket;
    }
}

sub connect {
    my ($self) = @_;
    my $socket;
    my $num_attempts = 0;

    while(!$self->is_connected) {
        if($num_attempts) {
            if($num_attempts >= $self->{max_connect_attempts}) {
                $self->log("ERROR: can't connect to cdict");
                exit(0);
            }

            $self->log("cdict: can't connect ($num_attempts attempts)");
            sleep(10);
        }
        $self->connect_once;
        $num_attempts++;
    }
}

sub disconnect {
    my ($self) = @_;

    close($self->{socket}) if $self->{socket};
    $self->{socket} = 0;

    $self->log("cdict: disconnected");
}

# вход: массив фраз
# выход: для каждой фразы кэшируется is_good_phrase
sub cache_good_phrases {
    my ($self, @phrases) = @_;
    @phrases = grep { ! defined($_->{cdict_is_good_phrase}) } @phrases;

    my @resp = $self->proj->datoteka_client->exec_cmds([map { [ 'getnorm', 'flags', $_ ] } @phrases]);
    for my $r (@resp) {
        my ($i, $resp) = @$r;
        my $phr = $phrases[$i];
        if(!defined $resp) {
            $phr->{cdict_is_good_phrase} = 1;
        } else {
            $phr->{cdict_is_good_phrase} = !($resp & $self->{bad_phrase_flag});
        }
    }

}

# вход: массив фраз
# выход: для каждой фразы кэшируется, широкая ли она
sub cache_wide_phrases {
    my ($self, @phrases) = @_;

    @phrases = grep { ! defined($_->{cdict_is_wide}) } @phrases;
    return () if !@phrases;

    my @resp = $self->proj->datoteka_client->exec_cmds([map { [ 'getnorm', 'flags', $_ ] } @phrases]);
    for my $h (@resp) {
        my ($i, $resp) = @$h;
        my $phr = $phrases[$i];
        if(!defined $resp) {
            $phr->{cdict_is_wide} = 0;
        } else {
            $phr->{cdict_is_wide} = $resp & $self->{wide_phrase_flag};
        }
    }
}

# вход: массив фраз
# выход: массив чисел (частот) с учётом языка
sub get_count_lang {
    my($self, $lang, @phrases) = @_;
    return () if !@phrases;

    my $resp = $self->exec_command(join("/", map{"get\tcount_$lang\t".$_->cdict_preprocess->norm_phr_uniq} @phrases));

    my @result = map{substr($_, 0, 2) eq "NO" ? 0 : substr($_, 4)} split "/", $resp;

    return @result;
}

# вход: массив фраз
# выход: массив пар (номер фразы, частота)
sub get_count {
    my($self, @phrases) = @_;
    my @resp = $self->exec_cmds([map { [ 'getnorm', 'count', $_ ] } @phrases]);
    $_->[1] //= 0 for @resp;
    return @resp;
}

# вход: массив фраз
# выход: массив чисел (частот)
sub get_count_generic {
    my $self = shift;
    my $type = shift;
    my @phrases = @_;
    return () if !@phrases;

    my $pack_size = 10000;
    my @result;
    while (@phrases) {
        my @pack = splice(@phrases, 0, $pack_size);
        my $cmd = join('/', map { join("\t", 'getnorm', $_->cdict_command($type), $_->cdict_preprocess_norm->text, $_->lang) } @pack);
        my $resp = $self->exec_command($cmd);
        push @result, map { substr($_, 0, 2) eq "NO" ? 0 : substr($_, 4) } split "/", $resp;
    }

    return @result;
}

# вход: массив фраз
# выход: массив пар (номер фразы, частота фразы как запроса)
sub get_query_count {
    my($self, @phrases) = @_;

    my @resp = $self->exec_cmds([map { [ 'getnorm', 'countq', $_ ] } @phrases]);
    $_->[1] //= 0 for @resp;
    return @resp;
}

# вход: массив фраз
# выход: массив пар (номер фразы, указатель на хэш вида "region_id => count")
sub get_regions_count {
    my($self, @phrases) = @_;

    # т.к. много данных, разбиваем на мелкие пачки
    my @resp = $self->exec_cmds([map { [ 'getnorm', 'countg', $_ ] } @phrases], pack_size => 10);
    for my $r (@resp) {
        my $item = $r->[1];
        if(!defined $item) {
            $r->[1] = {};
        } elsif(substr($item, 0, 1) eq "-") {
            $r->[1] = { "-" => 0 };
        } else {
            $r->[1] = { map{$_->[0] => $_->[1]} map{[split ":"]} grep{$_} split " ", $item };
        }
    }
    return @resp;
}

# вход: массив фраз
# выход: массив чисел (частот)
sub get_mobile_count {
    my($self, @phrases) = @_;
    my @resp = $self->exec_cmds([map { [ 'getnorm', 'countm', $_ ] } @phrases]);
    $_->[1] //= 0 for @resp;
    return @resp;
}


# вход: массив фраз
# выход: массив чисел (количество баннеров, в которых фраза встречается)
sub get_bnr_count {
    my($self, @phrases) = @_;
    return () if !@phrases;

    my $resp = $self->proj->datoteka_client->exec_command(join("/", map{join("\t", $self->{getnorm_cmd}, $_->cdict_command("bnr_count"), $_->cdict_preprocess_norm)} @phrases));

    my @result = map{substr($_, 0, 2) eq "NO" ? 0 : substr($_, 4)} split "/", $resp;

    return @result;
}

# вход: массив нормализованных текстов
# выход: массив чисел (частот)
sub get_normarr_count {
    my($self, $arr) = @_;
    return () if !@$arr;

    my $resp = $self->exec_command(join("/", map{"get\tcount\t".$_} @$arr));

    my @result = map{substr($_, 0, 2) eq "NO" ? 0 : substr($_, 4)} split "/", $resp;

    return @result;
}

# вход: массив фраз
# выход: у каждой фразы появляется поле {search_count}
sub cache_count {
    my($self, @phrases) = @_;
    return if !@phrases;

    my @counts = $self->get_count(@phrases);
    $phrases[$_]->{search_count} = $counts[$_] for 0..$#counts;
}

# вход: массив фраз
# выход: массив пар (номер фразы, PhraseList разваливания по синонимам)
sub get_syns {
    my($self, @phrases) = @_;

    my @resp = $self->proj->datoteka_client->exec_cmds([map { [ 'getsnorm', 'syns', $_ ] } @phrases]);
    for my $r (@resp) {
        my ($i, $resp) = @$r;
        my $texts = [];
        if(defined $resp) {
             $texts = [ split " , ", $resp ];
        }
        # добавляем главную синонимичную форму
        if(!grep{$_ eq $phrases[$i]->snorm_phr_uniq} @$texts) {
            push @$texts, $phrases[$i]->snorm_phr_uniq;
        }
        $r->[1] = $self->proj->phrase_list({ phrases_arr => $texts });
    }
    return @resp;
}

# вход: массив фраз
# выход: массив пар (номер фразы, ссылка на хэши вида "слово => частота")
sub get_tail {
    my($self, @phrases) = @_;

    my @resp = $self->exec_cmds([map { ['getnorm', 'tail', $_ ] } @phrases]);

    for my $r (@resp) {
        my $resp = $r->[1];
        if(!defined $resp) {
            $r->[1] = {};
        } else {
            my $h = {};
            for my $data (split " , ", $resp) {
                my @a = split " ", $data;
                $h->{$a[0]} = $a[-1];
            }
            $r->[1] = $h;
        }
    }
    return @resp;
}

# вход: массив фраз
# выход: к каждой фразе дописываются поля tail2count и tail2categs (tail => массив айдишников категорий)
sub cache_cdict_tail_categs {
    my ($self, @phrases) = @_;
    my @resp = $self->exec_cmds([map { [ 'getnorm', 'tail', $_ ] } @phrases]);
    for my $r (@resp) {
        my ($i, $resp) = @$r;
        my $phr = $phrases[$i];
        my $tail2categs = {};
        my $tail2count = {};

        if(defined $resp) {
            for my $data (split " , ", $resp) {
                my @a = split " ", $data;
                $tail2count->{$a[0]} = $a[-1];

                # категории сохраняем только для достаточно частотных слов
                next if $a[-1] <= 10;
                $tail2categs->{$a[0]} = [grep{$_} @a[1..($#a-1)]];
            }
        }

        $phr->{tail2count} = $tail2count;
        $phr->{tail2categs} = $tail2categs;
    }
}

sub get_id2cat :CACHE {
    my ($self) = @_;
    return $self->proj->categs_tree->ref_get_minicateg_by_id;
}

# вход: массив фраз
# выход: для каждой вразы кэшируются значения cdict_minicategs и cdict_atoms
sub cache_categs_atoms {
    my($self, @phrases) = @_;
    return () if !@phrases;

    my $resp = $self->proj->datoteka_client->exec_command(join("/", map{"get\tcategs\t".$_->cdict_preprocess->snorm_phr_uniq} @phrases));
    my @resp_arr = split "/", $resp;
    my $id2cat = $self->get_id2cat;

    for my $i (0..$#phrases) {
        if($resp_arr[$i] =~ /^NO/) {
            $phrases[$i]->{cdict_minicategs} = undef;
            $phrases[$i]->{cdict_atoms} = undef;
            next;
        }

        # ответ "ctg _ atom", но если категорий нет, то в ответе будет "_ atom1 atom2", или наоборот: "ctg1 ctg2 _"
        my ($categs, $atoms) = split /^_ | _ | _$/, substr($resp_arr[$i], 4), 2;
        $phrases[$i]->{cdict_minicategs} = [grep{$_} map{ $id2cat->($_) } split " ", $categs];
        $phrases[$i]->{cdict_atoms} = [grep{$_} map{ $id2cat->($_) } split " ", $atoms || ""];
    }
}

# вход: массив фраз
# выход: массив пар (номер фразы, ссылка на массивы категорий (или undef для фраз, которых нет в cdict))
sub get_categs {
    my($self, @phrases) = @_;
    my @resp = $self->proj->datoteka_client->exec_cmds([map { [ 'getsnorm', 'categs', $_ ] } @phrases]);
    my $id2cat = $self->get_id2cat;
    my @result;
    for my $r (@resp) {
        my $resp = $r->[1];
        if (!defined $resp) {
            $r->[1] = undef;
            next;
        }
        # ответ "ctg _ atom", но если категорий нет, то в ответе будет "_ atom1 atom2", или наоборот: "ctg1 ctg2 _"
        my ($categ_ids, $atom_ids) = split /^_ | _ | _$/, $resp, 2;
        $r->[1] = [ grep{ $_ } map{ $id2cat->($_) } split ' ', $categ_ids ];
    }
    return @resp;
}

# вход: массив фраз
# выход: у каждой фразы появляется поле cdict_regions_phrases вида "фраза => id региона"
sub cache_regions_phrases {
    my ($self, @phrases) = @_;
    my @resp = $self->proj->datoteka_client->exec_cmds([ map { ['get', 'regions', $_->cdict_key_snorm ] } @phrases ]);
    my $id2cat = $self->get_id2cat;
    my $phrase_index = 0;
    for my $r (@resp) {
        my ($i, $resp) = @$r;
        my $h = undef;
        if(defined $resp) {
            $h = {};
            for my $reg_phrase (split /\s*,\s*/, $resp) {
                my ($reg_id, $phrase) = $reg_phrase =~ /^([^\s]+)\s(.+)/;
                my $region = $id2cat->($reg_id);
                ($h->{$phrase} ||= {})->{$region}++ if $phrase && $region;
            }
        }
        $phrases[$i]->{cdict_regions_phrases} = $h;
    }
    return @resp;
}

# вход: массив фраз
# выход: массив ссылок на массивы атомов (или undef для фраз, которых нет в cdict)
sub get_atoms {
    my($self, @phrases) = @_;
    return () if !@phrases;

    my $resp = $self->proj->datoteka_client->exec_command(join("/", map{"get\tatoms\t".$_->cdict_preprocess->snorm_phr_uniq} @phrases));

    my $id2cat = $self->get_id2cat;
    my @result = map{/^NO/ ? undef : [grep{$_} map{ $id2cat->($_) } split " ", (substr $_, 4)]} split "/", $resp;

    return @result;
}
# вход: массив фраз
# выход: массив гармонизированных текстов
sub get_harmonized {
    my($self, @phrases) = @_;
    return () if !@phrases;

    my %h = $self->get_harmonized_hash(@phrases);

    my @result = map { defined $h{$_->norm_phr} ? $h{$_->norm_phr} : $_->norm_phr } @phrases;

    return @result;
}

# вход: массив фраз
# выход: хеш вида нормализованный текст => гармонизированный текст ( или undef, если фраза не гармонизируется )
sub get_harmonized_hash {
    my($self, @phrases) = @_;
    return () if !@phrases;

    my $resp = $self->proj->datoteka_client->exec_command(join("/", map{join("\t", "get", $_->cdict_command("harm"), $_->cdict_preprocess->norm_phr_uniq)} @phrases));
    my @resp_arr = split "/", $resp;
    my %result = map{ $phrases[$_]->norm_phr =>  $resp_arr[$_] =~ /^NO/ ? undef : substr($resp_arr[$_], 4)} 0..$#phrases;

    return %result;
}

# вход: массив фраз
# выход: у каждой фразы появляются поля-хэши "tail2count" и "tail2categs", массив "tails"
sub get_tail_categs {
    my($self, @phrases) = @_;
    return () if !@phrases;

    warn "Method 'get_tail_categs' is DEPRECATED, use 'cache_cdict_tail_categs' instead!";

    my $resp = $self->exec_command(join("/", map{"get\ttail\t".$_->cdict_preprocess->norm_phr_uniq} @phrases));

    my $i = 0;
    my $id2cat = $self->get_id2cat;
    for my $answ (split "/", $resp) {
        my $phr = $phrases[$i++];
        $phr->{tails} = [];
        $phr->{tail2count} = {};
        $phr->{tail2categs} = {};

        next if $answ =~ /^NO/;
        $answ =~ s/^YES\s//;

        for (split ",", $answ) {
            my ($word, @data) = split " ";
            my $count = $data[$#data];
            push @{$phr->{tails}}, $word;
            $phr->{tail2count}{$word} = $count;
            $phr->{tail2categs}{$word} = [grep{$_} map{$id2cat->($_)} @data[0..($#data-1)]];
        }
    }
}

# выполнение списка команд - новая версия
# каждая команда - список [cmd,ns,phr]
#   если cmd='getnorm' или 'getsnorm', то ключ определяется по объекту phr
#   если cmd='get', то phr - строка и нужно указать lang после phr
# возвращает список пар (i,resp), где:
#   i - номер запроса
#   resp - ответ, может быть undef, если данных нет в cdict
#   ответов может быть меньше, если не все данные были запрошены
# доп. параметры:
#   pack_size =>  макс. размер пачки для хождения с cdict-сервер (default: 10_000)
#
# на работу метода влияют настройки rpc ($proj->rpc->options):
#   local_cdict => $bool  -  если включено, работаем только с локальным кэшем
#   on_missing => $react  -  как реагировать на отсутсвующие в кэше значения
#       'store' - запомнить, что мы их запросили
#       'check' - проверить, что мы их запрашивали ранее
#       'skip'  - subj
#       'die'   - subj (бывш. DIE_ON_UNREQUESTED), this is default
sub exec_cmds {
    my $self = shift;
    my $cmds = shift;
    my %par  = (
        pack_size => 10_000,
        @_,
    );
    return () if !@$cmds;

    my $rpc = $self->proj->rpc;
    my $opt = $rpc->options;
    if ($opt->{local_cdict}) {
        my @resp;
        my $micro_cdict = $self->proj->{micro_cdict} // {};
        for my $i (0 .. $#$cmds) {
            my ($c, $ns, $phr, $lang) = @{$cmds->[$i]};
            my $key;
            # мы должны сами нормализовать, т.к. не ходим в сервер
            if ($c eq 'getnorm') {
                $key = $phr->cdict_key_norm;
            } elsif ($c eq 'getsnorm') {
                $key = $phr->cdict_key_snorm;
            } elsif ($c eq 'get') {
                $key = $phr;  # это строка
            } else {
                die "Cdict: Unsupported command: $c";
            }
            $lang //= (ref($phr) ? $phr->lang : '');
            $ns .= "_$lang" if $lang and $lang ne 'ru';
            my $cache = $micro_cdict->{$ns} // {};
            if (!exists $cache->{$key}) {
                my $old_reqs = $rpc->cdict_requests;
                my $react = $opt->{on_missing} // '';
                if ($react eq 'store') {
                    # режим $phl->cache.. методов, когда добавляются rpc
                    $old_reqs->{$ns}{$key} = 1;
                } elsif ($react eq 'check') {
                    # режим $phr->get.. методов в "простом" bmyt
                    if ($old_reqs->{$ns}{$key}) {
                        # значение было запрошено через cache.., кидаем штатное исключение
                        $rpc->die;
                    } else {
                        # забыли запросить, падаем
                        die "Cdict: Not requested: $c\t$ns\t$key\t$lang";
                    }
                } elsif ($react eq 'skip') {
                    # nothing to do
                } else {
                    # режим broad_mr (DIE_ON_UNREQUESTED)
                    die "Cdict: Not requested: $c\t$ns\t$key\t$lang";
                }
                next;
            }
            # $cache->{$key} может быть not defined, если был запрошен, но не найден в cdict-е
            # TODO что-то с этим сделать, чтобы избежать повторного кэширования в simple-режиме
            push @resp, [$i,$cache->{$key}];
        }
        return @resp;
    }

    # разбиваем на пачки
    my @todo = @$cmds;
    my @resp;
    my $i = 0;
    PACK: while (@todo) {
        # переводим в текстовый формат
        my @strs;
        my @cmds = splice(@todo, 0, $par{pack_size});
        for my $cmd (@cmds) {
            my ($c, $ns, $phr, $lang) = @$cmd;
            my $key;
            # нормализация будет на сервере, мы лишь минимально обрабатываем
            if ($c eq 'getnorm') {
                $key = $phr->cdict_preprocess_norm->text;
            } elsif ($c eq 'getsnorm') {
                $key = $phr->cdict_preprocess_norm->text;
            } elsif ($c eq 'get') {
                $key = $phr;  # это строка
            } else {
                die "Cdict: Unsupported command: $c";
            }
            $lang //= (ref($phr) ? $phr->lang : '');
            $ns .= "_$lang" if $lang and $lang ne 'ru';
            my $str;
            if ($c eq 'get') {
                $str = join("\t", $c, $ns, $key);
            } else {
                # отдельный $lang нужен для нормализации на сервере
                $str = join("\t", $c, $ns, $key, $lang);
            }
            push @strs, $str;
        }
        my $cmd = join('/', @strs);

        my $resp;
        my $attmt = 0;
        while(!defined($resp)) {
            $attmt++;
            if ($attmt > 100) {
                die "cdict: exec_command_once failed after $attmt attempts";
            }
            # реальное хождение в сервер
            $resp = $self->exec_command_once($cmd);
        }
        for my $r (split '/', $resp) {
            if (substr($r, 0, 2) eq 'NO') {
                push @resp, [$i++, undef];  # не найдено
            } else {
                push @resp, [$i++, substr($r, 4)];
            }
        }
    }
    return @resp;
}

sub exec_command {
    my ($self, $cmd) = @_;

    if ($ENV{LOCAL_CDICT}) {
        die "use exec_cmds in local cdict mode"
    }

    if(length($cmd) > 300000){
        my @arr = split "/", $cmd;
        my $limit = 10000;
        if(@arr > $limit){
            my @res = ();
            while( my @sarr = splice(@arr, 0, $limit) ){
                push(@res, $self->exec_command(join('/', @sarr)));
            }
            return join('/', @res);
        }
    }

    my $resp;
    my $attmt = 0;
    while(!defined($resp)) {
        $attmt++;
        if ($attmt > 100) {
            $self->log("ERROR: can't connect to cdict");
            last;
        }
        $resp = $self->exec_command_once($cmd);
    }
    return $resp;
}

sub get {
    # reqs = [[$ns1, $key1], [$ns2, $key2], ...]
    # return = [$val1 | undef, $val2 | undef, ... ]
    my ($self, $reqs) = @_;
    my @resp = $self->exec_cmds([map { [ 'get', @$_ ] } @$reqs ]);
    if (@resp != @$reqs) {
        die "CdictClient::get : not requested!\n";
    }
    return [ map { $_->[1] } @resp ];
}

sub exec_command_once {
    my ($self, $cmd) = @_;
    my $cmd_list = [ split /\//, $cmd ];

    my %cmdname_hash = map { (split /\t/)[0] => 1 } @$cmd_list;
    my @uncacheable_cmds = grep { !exists $cacheable_cmds{$_} } keys %cmdname_hash;

    my $resp_list;
    if ( @uncacheable_cmds ) {
        $resp_list = $self->exec_command_list_once( $cmd_list );
    }
    else {
        $resp_list = $self->exec_command_list_once_cached( $cmd_list );
    }

    if ( !defined $resp_list ) {
        return undef;
    }
    return join '/', @$resp_list;
}

sub exec_command_list_once_cached :LRUCACHELIST {
    my ($self, $cmd_list) = @_;
    return $self->exec_command_list_once( $cmd_list );
}

sub exec_command_list_once {
    my ($self, $cmd_list) = @_;
    my $cmd = join '/', @$cmd_list;
    my $num_bytes;

    $self->connect;

    if(!eval {
       $num_bytes = $self->{socket}->send("$cmd\n");
    }) {
        my $error = $@;
        $error =~ s/ERROR://g; # чтобы не паниковал мониторинг
        $self->log("send failed: $error");
        $num_bytes = undef;
    }

    if(!defined($num_bytes)) {
        $self->disconnect;
        $self->log("Socket::send() failed");
        return undef;
    }

    my @resp_chunks = ();
    while(!@resp_chunks || $resp_chunks[-1] !~ /\n/) {
        my $resp;
        my $timeout = $self->{can_read_timeout} || $MIN_TIMEOUT;
        my @ready = IO::Select->new($self->{socket})->can_read($timeout);
        if(!@ready) {
            $self->{can_read_timeout} = min($MAX_TIMEOUT, $TIMEOUT_DELTA + $timeout);
            $self->log("can_read timeout $timeout (query: '" . substr($cmd, 0, 50) ."')'");
            $self->disconnect;
            return undef;
        }

        if(!defined($self->{socket}->recv($resp, 1024)) || !$resp) {
            $self->disconnect;
            $self->log('WARNING on cdict ' . $self->host . ':' . $self->port . " - Socket::recv() failed on command [$cmd]");
            return undef;
        }

        $self->{can_read_timeout} = $MIN_TIMEOUT;
        push @resp_chunks, $resp;
    }

    my $full_resp = join "", @resp_chunks;
    chomp $full_resp;
    if ( $full_resp eq "BYE" ) {
        $self->disconnect;
        return undef;
    }
    if($full_resp =~ /^ERROR/) {
        $self->disconnect;
        $self->log("WARN: error in cdict full_resp: $full_resp");
        return undef;
    }

    return [split /\//, $full_resp, -1];
}

sub make_server {
    my $self = shift;
    # system("make -C ".$self->server_dir);
}

# По файлу с данными генерируем словарь и индекс
# Если задан флаг prepare_single_file, генерируем single_file
sub prepare_data_from_file {
    my ($self, $data_file, %prm) = @_;
    if ($prm{prepare_single_file}) {
        my $cdict_name = $data_file;
        $cdict_name =~ s/.*\///;

        my $single_tempfile = Utils::Sys::get_tempfile("$cdict_name\_temp\_file", DIR => $self->temp_dir, UNLINK => 1);
        my $cdict_gen_command = $self->server_dir."/cdict --mode=gen --data-file=$data_file --output-file=$single_tempfile";
        if ($prm{keep_raw_files} && $self->raw_files_dir) {
            -d $self->raw_files_dir || mkdir $self->raw_files_dir;
            $cdict_gen_command .= " --keep-raw-files --raw-files-dir=".$self->raw_files_dir;
        }
        $self->do_sys_cmd($cdict_gen_command);
        File::Copy::move($single_tempfile, $self->single_file) or die("move failed: $!");

        # сохранение архивных данных
        if ($prm{do_backup}) {
            my @now = localtime;
            my $suffix = sprintf(".%04d-%02d", 1900 + $now[5], 1 + $now[4]);
            my $backup_file_gzip = $self->single_file . $suffix . '.gz';
            unless (-e $backup_file_gzip) {
                $self->proj->log("create backup_file $backup_file_gzip ...");
                $self->proj->do_sys_cmd("gzip -c ".$self->single_file." > $single_tempfile");
                File::Copy::move($single_tempfile, $backup_file_gzip) or die("move failed: $!");
                $self->proj->log("create backup_file $backup_file_gzip done");
            }
        }
    } else {
        my $dict_tempfile = get_tempfile('dict', DIR => $self->temp_dir);
        my $index_tempfile = get_tempfile('index', DIR => $self->temp_dir);
        $self->do_sys_cmd($self->server_dir."/cdict --mode=gen-dict --data-file=$data_file --output-file=$dict_tempfile");
        $self->do_sys_cmd($self->server_dir."/cdict --mode=gen-index --data-file=$data_file --dict-file=$dict_tempfile --output-file=$index_tempfile");
        $self->do_sys_cmd("mv $dict_tempfile ".$self->dict_file);
        $self->do_sys_cmd("mv $index_tempfile ".$self->index_file);
    }
    return 1;
}

sub stop_server {
    my ($self) = @_;

    $self->connect_once;
    $self->exec_command("stop") if $self->is_connected;
}

1;
