package BM::BMClient::StaticMapClient;

use strict;

use utf8;
use open ':utf8';
no warnings 'utf8';
use base qw(ObjLib::ProjPart);

use IO::Select;
use Data::Dumper;
use Time::HiRes qw/gettimeofday tv_interval/;
use BM::Phrase;
use Utils::Common;
use Socket;

########################################################
# Интерфейс
########################################################

########################################################
# Инициализация
########################################################

__PACKAGE__->mk_accessors(qw(
    port
    host
    work_dir
    server_dir
    data_file
));

sub init {
    my ($self) = @_;

    $self->{socket} = 0;
    $self->{max_connect_attempts} = 500;
}

sub is_connected {
    my ($self) = @_;

    return $self->{socket};
}

sub connect_once {
    my ($self) = @_;

    return if $self->is_connected;

    my $socket = IO::Socket::INET->new(
        PeerAddr => inet_ntoa(inet_aton($self->host)), 
        PeerPort => $self->port, 
        Proto => "tcp",
        Timeout => 5
    );

    if(!$socket || !$socket->connected) {
        $socket = 0;

        $self->log("static_map: can't connect to ".$self->host.":".$self->port." ($!)");
    } else {
        binmode($socket, ":utf8");

        $self->log("static_map: connected to ".$self->host.":".$self->port);

        $self->{socket} = $socket;
    }
}

sub connect {
    my ($self) = @_;
    my $socket;
    my $num_attempts = 0;
    
    while(!$self->is_connected) {
        if($num_attempts) {
            if($num_attempts >= $self->{max_connect_attempts}) {
                $self->log("ERROR: can't connect to static_map");
                exit(0);
            }

            $self->log("static_map: can't connect ($num_attempts attempts)");
            sleep(10);
        }
        $self->connect_once;
        $num_attempts++;
    }
}

sub disconnect {
    my ($self) = @_;

    close($self->{socket}) if $self->{socket};
    $self->{socket} = 0;

    $self->log("static_map: disconnected");
}

sub exec_command {
    my ($self, $cmd) = @_;
    my $resp = undef;

    while(!defined($resp)) {
        $resp = $self->exec_command_once($cmd);
    }

    return $resp;
}

sub exec_command_once {
    my ($self, $cmd) = @_;
    my $num_bytes;

    $self->connect;
   
    if(!eval {
       $num_bytes = $self->{socket}->send("$cmd\n");
    }) {
        $self->log("send failed: $@");
        $num_bytes = undef;
    }

    if(!defined($num_bytes)) {
        $self->disconnect;
        $self->log("Socket::send() failed");
        return undef;
    }

    my @resp_chunks = ();
    while(!@resp_chunks || $resp_chunks[-1] !~ /\n/) {
        my $resp;

        my @ready = IO::Select->new($self->{socket})->can_read(5);
        if(!@ready) {
            $self->log("can_read timeout");
            $self->disconnect;
            return undef;
        }

        if(!defined($self->{socket}->recv($resp, 1024)) || !$resp) {
            $self->disconnect;
            $self->log("Socket::recv() failed");
            return undef;
        }

        push @resp_chunks, $resp;
    }

    my $full_resp = join "", @resp_chunks;
    chomp $full_resp;
    $self->disconnect if $full_resp eq 'session stopped' || $full_resp eq 'server stopped';
    if($full_resp =~ /^ERROR/) {
        $self->disconnect;
        return undef;
    }
    
    return $full_resp;
}

sub stop_server {
    my ($self) = @_;

    $self->connect_once;
    $self->exec_command("stop_server") if $self->is_connected;
}

sub get_value {
    my ($self, $key) = @_;

    my $query = "get_value\t$key\t";
    return $self->exec_command_once($query);
}

1;

