package BM::Banners::BannerFactory;
use strict;

use utf8;
use open ':utf8';

use std;
use base qw(ObjLib::ProjPart);

use Utils::Urls;
use DataSource::Elem;

use BM::Banners::Campaign;

use BM::Banners::LBannerBase;
use BM::Banners::LBannerAnalysis;
use BM::Banners::LBannerDirect;
use BM::Banners::LBannerBM;
use BM::Banners::LBannerProjSrv;
use BM::Banners::LBannerList;

########################################################
# Интерфейс
########################################################


########################################################
# Инициализация
########################################################

sub init {
    my ($self) = @_;
}

########################################################
# Методы
########################################################

# создание баннера по текстовой строке, в которой через "\t" записаны поля
# сначала поля banners, потом именованные поля ("name=val")
sub text2banner {
    my ($self, $text) = @_;
    return BM::Banners::LBannerBM->new($self->parse_text($text));
}

sub parse_text {
    my ($self, $text) = @_;
    chomp $text;
    my @data = split /\t/, $text, -1;

    # сначала берем поля banners
    my %data;
    my @fld = $self->get_string_fields();
    @data{@fld} = splice(@data, 0, scalar @fld);

    # костыль для добавления нового поля в banners -- не используем;
    # важно лишь, чтобы новый код использовал файлы с новым полем

    # оставшиеся поля, в том числе, поля banners_extended, с префиксом
    for my $field (@data) {
        my ($name, $val) = split /=/, $field, 2;
        $data{$name} = $val if defined $val;
    }
    $data{proj} = $self->proj;
    return \%data;
}

sub parse_text_tskv {
    my ($self, $text) = @_;
    chomp $text;

    my @fld = qw{ bid bannerid cid uid title body categories targetting phrases url active_flag bannerid bs_order_id lang_old site_domain group_id is_moderated image_flag status_show status_archived banner_type outer_category manual_category comment all_phrases phrase title_extension };
    my %data;
    for my $item (split /\t/, $text, -1) {
        my ($field, $value) = split /=/, $item, 2;
        $data{lc($field)} = $value if (($value) and ($field) and (grep { $_ =~ /^$field$/i } @fld));
    }

    my $bid = $data{id} || $data{bid} || 0;
    $data{id} = $data{bid} = $bid;

    # если среди полей есть bid или bannerid, но нет никаких других полей, относящихся к баннеру, по bid берём баннер
    my @banner_data_fld = qw{title title_extension body url phrases};
    my @data_fields = keys %data;
    my $need_to_load_bdata = (scalar (grep {$_ ~~ @banner_data_fld} @data_fields) == 0);

    if ($need_to_load_bdata) {
        # заполняем данные баннера из базы
        my $bnr;
        if (defined $data{bid}) {
            $bnr = $self->proj->bid2banner($data{bid});
        }
        elsif (defined $data{bannerid}) {
            $bnr = $self->proj->bsid2banner($data{bannerid});
        }
        if (defined $bnr) {
            # смогли загрузить баннер по bid, а можем и не смочь
            $data{$_} = $bnr->{$_} for @banner_data_fld;
            $data{all_phrases} = $data{phrases};
        }
    }
    elsif (exists $data{phrase}) {
        # берём из даных из файла
        $data{phrases} = $data{phrase};
        $data{all_phrases} = $data{phrase};
    }

    for my $field (@fld) {
        if (exists $data{$field}) {
            next;
        }
        $data{$field} = undef;
    }

    $data{proj} = $self->proj;
    return \%data;
}

sub get_string_fields {
    my $self = shift;
    return $self->mysql_banners_format;
}

sub campaign {
    my ($self, $data) = @_;
    $data->{proj} = $self->proj;
    return BM::Banners::Campaign->new($data);
}

sub lbanner {
    my ($self, $data) = @_;
    my $r = ref $data;
    if(! $r ){
        if( $data =~ /\d\t\d/ ){
            my $h = $self->parse_text($data);
            $h->{proj} = $self->proj;
            return BM::Banners::LBannerBM->new($h);
        }
    }elsif($r eq 'HASH'){
        my %hh = %$data;
        $hh{proj} = $self->proj;
        return BM::Banners::LBannerBM->new(\%hh);
    }elsif($r eq 'ARRAY'){
        my %hh = ();
        if (4 < scalar @$data) {
            @hh{qw{ title body url phrases region max_bid picture_url }} = @$data;
        } else {
            @hh{qw{ title body url phrases }} = @$data;
        }
        $hh{proj} = $self->proj;
        return BM::Banners::LBannerBM->new(\%hh);
    }else{
        return BM::Banners::LBannerBM->new({ proj => $self });
    }
}

sub psbanner {
    my ($self, $data) = @_;
    return $self->lbanner($data) unless $self->proj->use_projsrv;
    my $r = ref $data;
    if(! $r ){
        if( $data =~ /\d\t\d/ ){
            my $h = $self->parse_text($data);
            $h->{proj} = $self->proj;
            return BM::Banners::LBannerProjSrv->new($h);
        }
    }elsif($r eq 'HASH'){
        my %hh = %$data;
        $hh{proj} = $self->proj;
        return BM::Banners::LBannerProjSrv->new(\%hh);
    }elsif($r eq 'ARRAY'){
        my %hh = ();
        @hh{qw{ title body url phrases }} = @$data;
        $hh{proj} = $self->proj;
        return BM::Banners::LBannerProjSrv->new(\%hh);
    }else{
        return BM::Banners::LBannerProjSrv->new({ proj => $self });
    }
}

#@returns BM::Banners::LBannerList
sub banner_list {
    my ($self, $data) = @_;
    $data ||= [];
    my $r = ref $data;
    if($r eq 'ARRAY'){
        my (@sclrs, @objs) = ();
        ((! ref($_)) || (ref($_) =~ /^(HASH|ARRAY$)/)) ? push(@sclrs, $_) : push(@objs, $_) for @$data;
        my $bnl = BM::Banners::LBannerList->new( { banners_array => [ @objs, map { $self->lbanner($_) } @sclrs ], proj => $self->proj, lang => $self->proj->current_lang } );
        return $bnl;
    }else{
        return BM::Banners::LBannerList->new( { banners_array => [], proj => $self->proj, lang => $self->proj->current_lang } );
    }
}

sub banner_from_tskv {
    my ($self, $data) = @_;
    my $r = ref $data;
    if(! $r ){
        # это не ссылка, это просто строка. её можно сразу парсить
        my $h = $self->parse_text_tskv($data);
        $h->{proj} = $self->proj;
        $h->{lang} = $self->proj->current_lang;
        return BM::Banners::LBannerBM->new($h);
    }elsif($r eq 'HASH'){
        my %hh = %$data;
        $hh{proj} = $self->proj;
        $hh{lang} = $self->proj->current_lang;
        return BM::Banners::LBannerBM->new(\%hh);
    }elsif($r eq 'ARRAY'){
        my %hh = ();
        @hh{qw{ title body url phrases }} = @$data;
        $hh{proj} = $self->proj;
        $hh{lang} = $self->proj->current_lang;
        return BM::Banners::LBannerBM->new(\%hh);
    }else{
        return BM::Banners::LBannerBM->new({ proj => $self->proj, lang => $self->proj->current_lang });
    }
}

sub make_tskv {
    my $text = $_[0];

    if ((! $text) or (ref($text) ne 'ARRAY')) {
        return [];
    }

    my ($fields_str, @values_str) = @$text;

    my @fields = split /\t/, $fields_str;
    $_ =~ s/(^\s+)|(\s+$)//g for (@fields);
    my $size = scalar @fields;

    my @res = ();
    for my $line (@values_str) {
        my @values = split /\t/, $line;
        $_ =~ s/(^\s+)|(\s+$)//g for (@fields);
        push @res, (join "\t", map { "$fields[$_]=$values[$_]" } (0..($size-1)));
    }
    return \@res;
}

sub banner_list_from_file {
    my ($self, $data) = @_;

    if ((! $data) or (ref ($data) ne 'ARRAY') or (scalar @$data < 1)) {
        return BM::Banners::LBannerList->new( { banners_array => [], proj => $self->proj, lang => $self->proj->current_lang } );
    }

    my @file_strings = ();
    for my $s (@$data) {
        chomp $s;
        push @file_strings, $s;
    }

    # проверяем тип файла
    my $first_line = $file_strings[0];
    my @first_line_fields = split /\t/, $first_line;
    $_ =~ s/(^\s+)|(\s+$)//g for (@first_line_fields);

    my $tskv_strings = [];
    if (scalar (grep { $_ =~ /^[-a-zA-Z_]+$/ } @first_line_fields) == scalar (@first_line_fields)) {
        # считаем, что файл имеет такой формат: в первой строке поля через таб, в последующих - значения
        $tskv_strings = make_tskv($data);
    }
    elsif (scalar (grep { $_ =~ /^[-a-zA-Z_]+=.+$/ } @first_line_fields) == scalar (@first_line_fields)) {
        # считаем, что это tskv
        $tskv_strings = \@file_strings;
    }
    else {
        # файл в неправильном формате
        return BM::Banners::LBannerList->new( { banners_array => [], proj => $self->proj, lang => $self->proj->current_lang } );
    }

    my $bnl = BM::Banners::LBannerList->new( { banners_array => [ map { $self->banner_from_tskv($_) } @$tskv_strings ], proj => $self->proj, lang => $self->proj->current_lang } );
    return $bnl;
}

sub yt_banners_format {
    my $self = shift;
    my $columns_str = join(';', map {$_->{yt_field}} @{$self->proj->options->{yt_direct_banners_mapping}});
    return "<columns=[$columns_str];enable_escaping=false>schemaful_dsv";
}

sub mysql_banners_format {
    my $self = shift;
    my @fields =  map {$_->{mysql_field}} @{$self->proj->options->{yt_direct_banners_mapping}};
    return @fields;
}

1;
