package BM::Banners::BannerFactoryDB;
use strict;

use utf8;
use open ':utf8';

use std;
use Scalar::Util qw(weaken);
use base qw(BM::Banners::BannerFactory);

use Utils::Urls;
use DataSource::Elem;

use Utils::Words qw(word2norm);
use Utils::Sys qw(uniq);


__PACKAGE__->mk_accessors(qw(
    banners_dbh
    fields_mapping
));


my $BIDS_PER_REQUEST = 200_000;
my $CIDS_PER_REQUEST = 5;


sub init {
    my $self = shift;

    # по историческим причинам, часть полей в banners_extended в YT назвается не так, как в существовавшей раньше DirectBanners в mysql
    # в Utils::Common есть маппинг из одного в другое
    # во имя консистентности и совместимости со старым кодом, ниже при работе с banners_extended хардкодятся mysql-поля, а yt-поля получаются через маппинг

    my @fields_mapping = @{ $self->proj->options->{yt_direct_banners_mapping} };

    # костыль: сейчас в "общем" yt_banners_mapping нет поля title_extension, добавлять его нужно с осторожностью
    # поэтому пока добавляем его в маппинг, которым пользуется BannerFactoryDB
    push @fields_mapping, { yt_field => 'title_extension', mysql_field => 'title_extension' };

    $self->{fields_mapping}->{mysql2yt} = { map { $_->{mysql_field} => $_->{yt_field} } @fields_mapping };
    $self->{fields_mapping}->{yt2mysql} = { map { $_->{yt_field} => $_->{mysql_field} } @fields_mapping };
}

# из динтаблиц YT нельзя выбирать более миллиона строк разом: https://ml.yandex-team.ru/thread/yt/166351711236040622/
# некоторые методы (в духе "получить все баннеры для домена") склонны в этот лимит ударяться
# для таких случаев у нас имеется "альтернативная" BannerFactory на YQL - $proj->bf_slow - на которую мы делаем фоллбек
sub BF_SLOW_FALLBACK : ATTR(CODE) {
    my ($package, $symbol, $referent, $attr, $data, $phase, $filename, $linenum) = @_;

    no strict;
    no warnings 'redefine';

    # например: *BM::Banners::BannerFactoryDB::domains2banners => 'domains2banners'
    my ($method_name) = scalar(*$symbol) =~ m/::([^\:]+)$/;

    *$symbol = sub {
        my $self = shift;

        my @result = eval { $referent->($self, @_) };
        if ($@ && $@ =~ m/Query terminated prematurely due to excessive input/) {
            $self->proj->log(__PACKAGE__ . ": method '$method_name' failed, fallback to " . ref($self->proj->bf_slow));
            return $self->proj->bf_slow->$method_name(@_);
        } else {
            return (scalar(@result) == 1) ? $result[0] : @result;
        }
    };
}

# общий метод для преобразования id
# на входе:
#   $ids_arr  -  ссылка на массив айди
#   from      => входной id ('bid', 'BannerID', 'cid', ...)
#   to        => выходной id
#   hash            => 0|1  возвращать не список, а href (from => to)
#   reverse_hash    => 0|1  возвращать не список, а href (наоборот, to => from)
#   pack_size => размер пачки; по умолчанию: 20_000
#   verbose   => пишем в лог про каждую пачку
sub convert_ids {
    my $self = shift;
    my $ids = shift;
    my %par = @_;

    my $from = $par{from} // die "Define 'from' id field!";
    my $to = $par{to} // die "Define 'to' id field!";
    my $pack_size = $par{pack_size} // 20_000;
    my $return_hash = $par{hash};
    my $return_reverse_hash = $par{reverse_hash};

    die "Can't convert '$from' to itself" if $from eq $to;
    die "It's either hash or reverse hash" if $return_hash && $return_reverse_hash;

    my $mysql2yt = $self->fields_mapping->{mysql2yt};
    $from = $mysql2yt->{$from} if defined $mysql2yt->{$from};
    $to = $mysql2yt->{$to} if defined $mysql2yt->{$to};

    my $res = {};
    my @ids = sort { $a <=> $b } uniq(@$ids);
    my ($ids_done, $packs_done) = (0, 0);
    while (@ids) {
        my @pack = splice(@ids, 0, $pack_size);
        my $ids_list = join(',', @pack);

        my $rows = $self->proj->dt_proxy_client->do_select(
            table => $self->get_banners_dyntable(key_field => $from),
            fields => [$from, $to],
            condition => "$from in ($ids_list)",
            slice_result => 0,
        );

        if ($return_hash) {
            $res->{$_->[0]} = $_->[1] for @$rows;
        } elsif ($return_reverse_hash) {
            $res->{$_->[1]} = $_->[0] for @$rows;
        } else {
            $res->{$_->[1]} = 1 for @$rows;
        }

        $packs_done++;
        $ids_done += scalar(@pack);

        if ($par{verbose}) {
            $self->log("convert_ids: [$packs_done] $ids_done ids processed");
        }
    }

    if ($par{verbose}) {
        $self->log("convert_ids: all done!");
    }

    if ($par{hash} or $par{reverse_hash}) {
        return $res;
    } else {
        return keys %$res;
    }
}

# преобразование одного айдишника (force hash=0)
sub convert_id {
    my $self = shift;
    my $id = shift;
    my %par = @_;

    my @ids = $self->convert_ids([$id], %par, hash => 0);
    return $ids[0];
}

sub bid2cid {
    my $self = shift;
    my $bid  = shift;
    return $self->convert_id($bid, from => 'bid', to => 'cid');
}

sub get_banners_dyntable {
    my $self = shift;
    my %opts = @_;

    my $key_field = $opts{key_field};
    my $table = '//home/catalogia/dyntables/resources/banners_extended';

    return ($key_field && $key_field ne 'bid') ? "${table}_by_${key_field}" : $table;
}

sub get_campaigns_table : CACHE {
    my ($self) = @_;
    my $ds = DataSource::Elem->new({ proj => $self->proj, db_table => 'Campaigns', id_field => 'cid' });
    $ds->dbh($self->{'banners_dbh'});
    return $ds;
}

sub dbhash2banner {
    my ($self, $el) = @_;
    my %h = %$el;
    $h{'proj'} = $self->proj;

    my $lang = delete $h{lang};
    $h{lang_old} //= $lang;

    my $bnr = $self->proj->use_projsrv ? $self->psbanner(\%h) : $self->lbanner(\%h);
    return $bnr;
}

sub ythash2banner {
    my ($self, $el) = @_;
    my $el_db = { map { ( $self->fields_mapping->{yt2mysql}{$_} || $_) => $el->{$_} } keys $el };
    return $self->dbhash2banner($el_db);
}

sub ythashes2bnl {
    my ($self, $ythashes) = @_;
    return $self->banner_list([
            map { $self->ythash2banner($_) } @$ythashes
    ]);
}

sub get_banner_dbhashes {
    my $self = shift,
    my %opts = @_;

    my $condition = $opts{condition};
    my $key_condition = $opts{key_condition};
    my $key_field = $opts{key_field};

    die "provide some condition" unless $condition || ($key_field && $key_condition);

    my $mapping = $self->fields_mapping->{yt2mysql};
    if ($key_field && $key_field ne 'bid') {
        return $self->proj->dt_proxy_client->do_select_by_key(
            table => $self->get_banners_dyntable(),
            id_field => 'bid',
            key_field => $key_field,
            fields => [ keys %$mapping ],
            fields_mapping => $mapping,
            condition => $condition,
            key_condition => $key_condition,
        );
    } else {
        my $combined_condition;
        if (defined($condition) && defined($key_condition)) {
            $combined_condition = "($key_condition) and ($condition)";
        } else {
            $combined_condition = $key_condition // $condition;
        }

        return $self->proj->dt_proxy_client->do_select(
            table => $self->get_banners_dyntable(),
            fields => [ keys %$mapping ],
            fields_mapping => $mapping,
            condition => $combined_condition,
        );
    }
}

sub _ids2bnrs {
    my $self = shift;
    my $bids = shift // die "provide bids";

    my @bids = uniq(@$bids);
    return $self->banner_list([]) if !@bids;

    my @res;
    while (my @batch = splice(@bids, 0, $BIDS_PER_REQUEST)) {
        push @res, @{ $self->get_banner_dbhashes(
            condition => "bid in (" . join(", ", @batch) . ")",
        ) };
    }

    return \@res;
}

sub ids2bnl {
    my $self = shift;
    my $bids = shift;
    my $bnrs = $self->_ids2bnrs($bids);
    return $self->banner_list([map { $self->dbhash2banner($_) } @$bnrs]);
}

# Obsolete
sub _ids2bnl {
    my ($self, @args) = @_;
    $self->ids2bnl(@args);
}

#@returns BM::Banners::LBannerList
sub _bsids2bnl {
    my ($self, $lst) = @_;

    my @bs_banner_ids = uniq grep { $_ } @$lst;
    my $key_field = $self->fields_mapping->{mysql2yt}->{bs_banner_id};

    my @dbhashes;
    while (my @chunk = splice(@bs_banner_ids, 0, $BIDS_PER_REQUEST)) {
        push @dbhashes, @{ $self->get_banner_dbhashes(
            key_field => $key_field,
            key_condition => "$key_field in (" . join(",", @chunk) . ")"
        ) };
    }

    return $self->banner_list([
        map { $self->dbhash2banner($_) } @dbhashes
    ]);
}

sub List_SQL {
    my ($self, $SQL) = @_;
    my $ds = DataSource::Elem->new({ proj => $self->proj });
    $ds->dbh($self->{'banners_dbh'});
    return $ds->List_SQL($SQL);
}

sub get_campaign_banners {
    my ($self, $cid) = @_;

    my @bids = $self->convert_ids([$cid], from => 'cid', to => 'bid');
    my $bnl = $self->ids2bnl(\@bids);
    return [ @$bnl ];
}

sub lbanner {
    my ($self, $data) = @_;
    if (!ref($data) && ($data =~ /^\s*\d+\s*,?\s*$/)) {
        my $bnl = $self->_ids2bnl([ $data ]);
        return undef unless $bnl->count;
        return $bnl->[0];
    } else {
        return $self->SUPER::lbanner($data);
    }
}

#@returns BM::Banners::LBannerList
sub banner_list {
    my ($self, $data) = @_;
    $data ||= [];
    if (!ref($data) && ($data =~ /^\s*\d+\s*(,\s*\d+\s*)*$/)) {
        return $self->_ids2bnl([split(',', $data)]);
    } elsif (ref($data) eq 'ARRAY') {
        my (@ids, @others) = ();
        (!ref($_) && ($_ =~ /^\s*\d+\s*,?\s*$/)) ? push(@ids, $_) : push(@others, $_) for @$data;
        if (@ids) {
            return $self->_ids2bnl(\@ids) + $self->SUPER::banner_list(\@others) if @others;
            return $self->_ids2bnl(\@ids);
        }
        return $self->SUPER::banner_list($data);
    } else {
        return $self->SUPER::banner_list($data);
    }
}

sub get_banner_by_id {
    my ($self, $bid) = @_;
    my $bnl = $self->_ids2bnl([$bid]);
    return (@$bnl)[0];
}

#@returns BM::Banners::LBannerBM
sub get_banner_by_bsid {
    my ($self, $BannerID) = @_;
    my $bnl = $self->_bsids2bnl([$BannerID]);
    return (@$bnl)[0];
}

sub campaign {
    my ($self, $data) = @_;
    my $rf = ref($data);
    if(!ref($data) && ($data =~ /^\s*\d+\s*,?\s*$/)) {
        return $self->SUPER::campaign({ campaign_id => $data });
    } else {
        return $self->SUPER::campaign($data);
    }
}

sub get_campaign_inf {
    my ($self, $cmp) = @_;
    return $self->get_campaigns_table->List2( filter => {cid => $cmp} );
}

sub _get_campaign_dbhashes {
    my ($self, $list) = @_;
    $list = [ ref($list) ? @$list : $list ];

    my $key_field = $self->fields_mapping->{mysql2yt}->{campaign_id};

    my @res;
    while (my @chunk = splice(@$list, 0, $CIDS_PER_REQUEST)) {
        my $chunk_res = $self->get_banner_dbhashes(
            key_field => $key_field,
            key_condition => "$key_field in (" . join(",", @chunk) . ")",
        );

        push @res, @$chunk_res;
    }

    return \@res;
}

sub get_campaign_phrases {
    my ($self, $list) = @_;

    my $res = $self->_get_campaign_dbhashes($list);
    return uniq map { $_->{'phrases'} } @$res;
}

sub get_campaign_banner_texts {
    my ($self, $list) = @_;

    my $res = $self->_get_campaign_dbhashes($list);
    return map { $_->{'title'}.'    '.$_->{'body'} } @$res;
}

sub get_domain_phrases : BF_SLOW_FALLBACK {
    my ($self, $domains, %prm) = @_;

    my $key_field = $self->fields_mapping->{mysql2yt}->{site_domain};
    my $res = $self->get_banner_dbhashes(
        key_field => $key_field,
        key_condition => "$key_field in (" . join(",", map { "'$_'" } @$domains) . ")",
    );

    # todo: использовать "настоящий" limit
    @$res = splice(@$res, 0, $prm{limit}) if $prm{limit};

    return map {[$_->{'site_domain'}, $_->{'phrases'}]} @$res;
}

sub get_campaigns_by_domain : BF_SLOW_FALLBACK {
    my ($self, $domains) = @_;
    $domains = [$domains] unless ref($domains);

    return uniq $self->convert_ids(
        [ map { "'$_'" } @$domains ],
        from => "site_domain",
        to => "campaign_id",
    );
}

sub get_banners_by_bsbannerid {
    my ($self, $lst) = @_;
    return $self->convert_ids($lst, from => 'BannerID', to => 'bid');
}

sub campaigns2banners {
    my ($self, $cids) = @_;
    $self->convert_ids($cids, from => 'cid', to => 'bid', pack => $CIDS_PER_REQUEST);
}

sub campaigns2orderids {
    my ($self, $cmps) = @_;
    $self->proj->big_sql_data;
    my $res = $self->get_campaigns_table->List2( filter => {'cid' => $cmps}, gfields => ['OrderID'] );
    return map { $_->{OrderID} } @$res;
}

sub campaigns2groupid {
    my ($self, $cmps) = @_;
    $cmps = [$cmps] unless ref($cmps);

    return uniq $self->convert_ids(
        $cmps,
        from => "campaign_id",
        to => "group_id",
        pack_size => $CIDS_PER_REQUEST,
    );
}

sub campaigns2campaigns_domains {
    my ($self, $list) = @_;

    my $res = $self->_get_campaign_dbhashes($list);
    return uniq map { $_->{campaign_id} . ' =*> ' . $_->{site_domain} } @$res;
}

sub banners2campaigns {
    my ($self, $bids) = @_;
    $self->convert_ids($bids, from => 'bid', to => 'cid');
}

sub banners2activebanners {
    my ($self, $lst) = @_;
    $lst = [$lst] unless ref($lst);

    my $id_field = $self->fields_mapping->{mysql2yt}->{id};
    my $active_flag = $self->fields_mapping->{mysql2yt}->{active_flag};

    my $res = $self->proj->dt_proxy_client->do_select(
        table => $self->get_banners_dyntable(),
        fields => [ $id_field ],
        condition => "$id_field in (" . join(",", @$lst) . ") and ($active_flag = 1)",
        slice_result => 0,
    );

    return map { $_->[0] } @$res;
}

sub domains2campaigns {
    my ($self, $dmns) = @_;
    return $self->get_campaigns_by_domain($dmns);
}

sub _get_firstcampaigns : BF_SLOW_FALLBACK {
    my ($self, $dmns) = @_;

    my $uid_field = $self->fields_mapping->{mysql2yt}->{uid};
    my $domain_field = $self->fields_mapping->{mysql2yt}->{site_domain};
    my $campaign_id_field = $self->fields_mapping->{mysql2yt}->{campaign_id};

    my $res = $self->proj->dt_proxy_client->do_select(
        table => $self->get_banners_dyntable(key_field => $domain_field),
        fields => [ $uid_field, $domain_field, $campaign_id_field ],
        fields_mapping => $self->fields_mapping->{yt2mysql},
        condition => "$domain_field in (" . join(",", map { "'$_'" } @$dmns) . ")",
    );

    my $first_campaigns = {};
    for my $row (@$res) {
        my $uid = $row->{uid};
        if (
            !defined($first_campaigns->{$uid})
            || ($first_campaigns->{$uid}->{campaign_id} > $row->{campaign_id})
        ) {
            $first_campaigns->{$uid} = $row;
        }
    }

    return $first_campaigns;
}

sub domains2logins_firstcampaign  {
    my ($self, $dmns) = @_;
    $dmns = [$dmns] unless ref($dmns);

    my $first_campaigns = $self->_get_firstcampaigns($dmns);
    my $login_hash = $self->proj->get_uid2login_hash([ keys %$first_campaigns ]);

    return map {
        $_->{site_domain} . ' =*> ' . $login_hash->{$_->{uid}} . ' ' . $_->{campaign_id}
    } sort {
        $b->{campaign_id} <=> $a->{campaign_id}
    } values %$first_campaigns;
}

sub domains2domains_login : BF_SLOW_FALLBACK {
    my ($self, $dmns) = @_;
    $dmns = [$dmns] unless ref($dmns);

    my $uid_field = $self->fields_mapping->{mysql2yt}->{uid};
    my $domain_field = $self->fields_mapping->{mysql2yt}->{site_domain};

    my $res = $self->proj->dt_proxy_client->do_select(
        table => $self->get_banners_dyntable(key_field => $domain_field),
        fields => [ $uid_field, $domain_field ],
        fields_mapping => $self->fields_mapping->{yt2mysql},
        condition => "$domain_field in (" . join(",", map { "'$_'" } @$dmns) . ")",
    );

    my $login_hash = $self->proj->get_uid2login_hash([ uniq map { $_->{uid} } @$res ]);

    return uniq map { $_->{site_domain} . ' =*> ' . $login_hash->{$_->{uid}} } @$res;
}

sub domains2banners : BF_SLOW_FALLBACK {
    my ($self, $dmns) = @_;
    $dmns = [$dmns] unless ref($dmns);

    my $id_field = $self->fields_mapping->{mysql2yt}->{id};
    my $domain_field = $self->fields_mapping->{mysql2yt}->{site_domain};

    my $res = $self->proj->dt_proxy_client->do_select(
        table => $self->get_banners_dyntable(key_field => $domain_field),
        fields => [ $id_field ],
        condition => "$domain_field in (" . join(",", map { "'$_'" } @$dmns) . ")",
        slice_result => 0,
    );

    return map { $_->[0] } @$res;
}

# todo: добавить в banners_extended нужное поле и переписать на запросы к dt_proxy, либо выкинуть совсем
# BannerReverseDomain похоже что уже давно не обновляется
sub domains2subdomains {
    my ($self, $dmns) = @_;
    my @arr = ();
    for my $d (@$dmns){
        my $res = $self->List_SQL("
            select reverse(reverse_site_domain) site_domain, count(*) cc
            from BannerReverseDomain
            where reverse_site_domain like reverse('\%.$d')
            group by reverse_site_domain
        ");
        push(@arr, map { $_->{'site_domain'}.' => '.$_->{'cc'} } @$res);
    }
    return @arr;
}

sub bsbannerid2banners {
    my ($self, $lst) = @_;
    return $self->get_banners_by_bsbannerid($lst);
}

sub banners2bsbannerid {
    my ($self, $lst) = @_;
    $lst = [$lst] unless ref($lst);

    return uniq $self->convert_ids($lst, from => 'id', to => 'bs_banner_id');
}

sub logins2campaigns {
    my ($self, $lst) = @_;
    $self->proj->big_sql_data;
    return () unless @$lst;
    my $l2u = $self->proj->get_login2uid_hash($lst);
    my @arr = keys %{{ map { $_=>1 } grep {$_} values %$l2u }};
    return () unless @arr;
    my $res = $self->get_campaigns_table->List2( filter => {'uid' => \@arr}, gfields => ['cid', 'uid'], );
    return map { $_->{'cid'} } @$res;
}

sub clientids2logins {
    my ($self, $lst) = @_;
    $self->proj->big_sql_data;
    return () unless @$lst;
    my $res = $self->get_campaigns_table->List2( filter => {'ClientID' => $lst}, gfields => ['ClientID', 'uid'], );
    my $u2l = $self->proj->get_uid2login_hash([ map { $_->{'uid'} } @$res ]);
    my @arr = keys %{{ map { $_=>1 } grep {$_} values %$u2l }};
    return @arr;
}

sub clientids2clientids_logins {
    my ($self, $lst) = @_;
    $self->proj->big_sql_data;
    return () unless @$lst;
    my $res = $self->get_campaigns_table->List2( filter => {'ClientID' => $lst}, gfields => ['ClientID', 'uid'], );
    my $u2l = $self->proj->get_uid2login_hash([ map { $_->{'uid'} } @$res ]);
    my @arr = keys %{{ map { $_->{ClientID}." ".$u2l->{$_->{uid}} => 1 } @$res }};
    return @arr;
}

sub clientids2campaigns {
    my ($self, $lst) = @_;
    $self->proj->big_sql_data;
    return () unless @$lst;
    my $res = $self->get_campaigns_table->List2( filter => {'ClientID' => $lst}, gfields => ['ClientID', 'cid'], );
    return map { $_->{'cid'} } @$res;
}

sub campaigns2logins {
    my ($self, $lst) = @_;
    $self->proj->big_sql_data;
    return () unless @$lst;
    my $res = $self->get_campaigns_table->List2( filter => {'cid' => $lst}, gfields => ['cid', 'uid'], );
    my $u2l = $self->proj->get_uid2login_hash([ map { $_->{'uid'} } @$res ]);
    my @arr = keys %{{ map { $_=>1 } grep {$_} values %$u2l }};
    return @arr;
}

sub campaigns2campaigns_names {
    my ($self, $lst) = @_;
    $self->proj->big_sql_data;
    return () unless @$lst;
    my $res = $self->get_campaigns_table->List2( filter => {'cid' => $lst}, gfields => ['cid', 'name'], );
    my @arr = map { $_->{cid}." ".$_->{name} } @$res;
    return @arr;
}

sub campaigns2clientids {
    my ($self, $lst) = @_;
    $self->proj->big_sql_data;
    return () unless @$lst;
    my $res = $self->get_campaigns_table->List2( filter => {'cid' => $lst}, distinct => ['ClientID'], );
    my @arr = map { $_->{'ClientID'} } @$res;
    return @arr;
}

sub logins2clientids {
    my ($self, $lst) = @_;
    $self->proj->big_sql_data;
    return () unless @$lst;
    my $l2u = $self->proj->get_login2uid_hash($lst);
    my @arr = keys %{{ map { $_=>1 } grep {$_} values %$l2u }};
    return () unless @arr;
    my $res = $self->get_campaigns_table->List2( filter => {'uid' => \@arr}, distinct => ['ClientID'], );
    return map { $_->{'ClientID'} } @$res;
}

sub logins2uids {
    my ($self, $lst) = @_;
    $self->proj->big_sql_data;
    return () unless @$lst;
    my $l2u = $self->proj->get_login2uid_hash($lst);
    my @arr = keys %{{ map { $_=>1 } grep {$_} values %$l2u }};
    return @arr;
}

sub logins2logins_uids {
    my ($self, $lst) = @_;
    $self->proj->big_sql_data;
    return () unless @$lst;
    my $l2u = $self->proj->get_login2uid_hash($lst);
    my @arr = map { "$_ ".$l2u->{$_} }  keys %$l2u;
    return @arr;
}

sub logins2logins_clientids {
    my ($self, $lst) = @_;
    $self->proj->big_sql_data;
    return () unless @$lst;
    my $l2u = $self->proj->get_login2uid_hash($lst);
    my @arr = keys %{{ map { $_=>1 } grep {$_} values %$l2u }};
    return () unless @arr;
    my $u2l = { map { $l2u->{$_} => $_ } keys %$l2u };
    my $res = $self->get_campaigns_table->List2( filter => {'uid' => \@arr}, distinct => ['uid', 'ClientID'], );
    return map { $u2l->{$_->{uid}}.' '.$_->{'ClientID'} } @$res;
}

sub logins2logins_domains {
    my ($self, $lst) = @_;
    $self->proj->big_sql_data;
    return () unless @$lst;
    my $l2u = $self->proj->get_login2uid_hash($lst);
    my $u2l = { map { $l2u->{$_} => $_ } keys %$l2u };
    my @arr = keys %{{ map { $_=>1 } grep {$_} values %$l2u }};
    return () unless @arr;

    my $cmps = [ map { $_->{cid} } @{ $self->get_campaigns_table->List2( filter => {'uid' => \@arr}, gfields => ['cid', 'uid'], ) } ];
    my $res = $self->_get_campaign_dbhashes($cmps);

    my $goodlogins = { map { $u2l->{$_->{uid}} => 1 } @$res };
    return (map { $u2l->{ $_->{uid} }.' =*> '.$_->{'site_domain'} } @$res), ( grep { ! $goodlogins->{$_} } @$lst );
}

sub random_banners {
    my ($self, $cnt, $lang) = @_;
    $cnt ||= 10;

    my $prms = {};
    $prms->{lang} = [ $lang ] if $lang;
    my $lst = [ split ',', $self->proj->random_banners_client->k_random_banners($cnt, $prms) ];
    return $self->_ids2bnl($lst);
}

sub random_active_banners {
    my ($self, $cnt, $lang) = @_;
    $cnt ||= 10;

    my $prms = { active_flag => [1] };
    $prms->{lang} = [ $lang ] if $lang;
    my $lst = [ split ',', $self->proj->random_banners_client->k_random_banners($cnt, $prms) ];
    return $self->_ids2bnl($lst);
}

1;
